<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Tymon\JWTAuth\Exceptions\JWTException;
use Tymon\JWTAuth\Facades\JWTAuth;
use Illuminate\Support\Facades\Auth;
use Tymon\JWTAuth\Exceptions\TokenExpiredException;
use App\Models\User;
use OpenApi\Attributes as OA;

#[OA\Tag(name: 'Autenticación', description: 'Gestión de autenticación y tokens JWT')]
class AuthController extends Controller
{
    #[OA\Post(
        path: '/api/auth/login',
        summary: 'Iniciar sesión',
        description: 'Autentica un usuario y genera un token JWT',
        requestBody: new OA\RequestBody(
            required: true,
            content: new OA\JsonContent(
                required: ['email', 'password'],
                properties: [
                    new OA\Property(property: 'email', type: 'string', format: 'email', example: 'usuario@ejemplo.com'),
                    new OA\Property(property: 'password', type: 'string', format: 'password', example: 'password123'),
                ]
            )
        ),
        tags: ['Autenticación'],
        responses: [
            new OA\Response(
                response: 200,
                description: 'Login exitoso',
                content: new OA\JsonContent(
                    properties: [
                        new OA\Property(property: 'status', type: 'boolean', example: true),
                        new OA\Property(
                            property: 'data',
                            type: 'object',
                            properties: [
                                new OA\Property(property: 'id', type: 'integer', example: 1),
                                new OA\Property(property: 'name', type: 'string', example: 'Juan Pérez'),
                                new OA\Property(property: 'email', type: 'string', example: 'usuario@ejemplo.com'),
                            ]
                        ),
                        new OA\Property(property: 'token', type: 'string', example: 'eyJ0eXAiOiJKV1QiLCJhbGc...'),
                        new OA\Property(property: 'expires_in', type: 'integer', example: 3600),
                        new OA\Property(property: 'token_type', type: 'string', example: 'Bearer'),
                    ]
                )
            ),
            new OA\Response(
                response: 401,
                description: 'Credenciales inválidas',
                content: new OA\JsonContent(
                    properties: [
                        new OA\Property(property: 'status', type: 'boolean', example: false),
                        new OA\Property(property: 'message', type: 'string', example: 'Invalid credentials'),
                    ]
                )
            ),
            new OA\Response(
                response: 500,
                description: 'Error al crear token',
                content: new OA\JsonContent(
                    properties: [
                        new OA\Property(property: 'status', type: 'boolean', example: false),
                        new OA\Property(property: 'message', type: 'string', example: 'Could not create token'),
                    ]
                )
            ),
        ]
    )]
    public function login(Request $request)
    {
        $credentials = $request->only('email', 'password');

        try {
            if (!$token = JWTAuth::attempt($credentials)) {
                return response()->json([
                    'status' => false,
                    'message' => 'Invalid credentials'
                ], 401);
            }
        } catch (JWTException $e) {
            return response()->json([
                'status' => false,
                'message' => 'Could not create token'
            ], 500);
        }

    $centralUser = JWTAuth::setToken($token)->toUser();
        tenancy()->initialize($centralUser->tenant_id);
        $tenantUser = User::find($centralUser->id);

        return response()->json([
            'status' => true,
            'data' => $tenantUser,
            'token' => $token,
            'expires_in' => JWTAuth::factory()->getTTL() * 60,
            'token_type' => 'Bearer',
        ]);
    }

    #[OA\Get(
        path: '/api/auth/logout',
        summary: 'Cerrar sesión',
        description: 'Invalida el token JWT del usuario autenticado',
        security: [['bearerAuth' => []]],
        tags: ['Autenticación'],
        responses: [
            new OA\Response(
                response: 200,
                description: 'Sesión cerrada exitosamente',
                content: new OA\JsonContent(
                    properties: [
                        new OA\Property(property: 'message', type: 'string', example: 'Successfully logged out'),
                    ]
                )
            ),
            new OA\Response(
                response: 500,
                description: 'Error al cerrar sesión',
                content: new OA\JsonContent(
                    properties: [
                        new OA\Property(property: 'error', type: 'string', example: 'Failed to logout, please try again'),
                    ]
                )
            ),
        ]
    )]
    public function logout(Request $request)
    {
        try {
            JWTAuth::invalidate(JWTAuth::getToken());
        } catch (JWTException $e) {
            return response()->json(['error' => 'Failed to logout, please try again'], 500);
        }

        return response()->json(['message' => 'Successfully logged out']);
    }

    #[OA\Get(
        path: '/api/auth/me',
        summary: 'Obtener información del usuario autenticado',
        description: 'Devuelve los datos del usuario autenticado y su empresa',
        security: [['bearerAuth' => []]],
        tags: ['Autenticación'],
        responses: [
            new OA\Response(
                response: 200,
                description: 'Información del usuario obtenida exitosamente',
                content: new OA\JsonContent(
                    properties: [
                        new OA\Property(property: 'user', type: 'object'),
                        new OA\Property(property: 'empresa', type: 'object'),
                    ]
                )
            ),
            new OA\Response(
                response: 401,
                description: 'No autorizado',
                content: new OA\JsonContent(
                    properties: [
                        new OA\Property(property: 'error', type: 'string', example: 'Sin autorización para acceder al recurso'),
                    ]
                )
            ),
        ]
    )]
    public function me(Request $request)
    {
        $user = auth('api')->user();

        // Obtener empresa desde la base de datos central (sólo las columnas relevantes)
        try {
            $query = \App\Models\CatEmpresa::on('mysql')
                ->select('idempresa', 'rfc', 'razon_social', 'telefono1', 'telefono2', 'email', 'activo', 'comentarios');

            // Si el usuario autenticado trae tenant_id en el token, usarlo para filtrar
            if (!empty($user) && isset($user->tenant_id)) {
                $query = $query->where('idempresa', $user->tenant_id);
            }

            $empresa = $query->first();
        } catch (\Exception $e) {
            $empresa = null;
        }

        return response()->json([
            'user' => $user,
            'empresa' => $empresa ? $empresa->toArray() : null
        ]);
    }

    #[OA\Get(
        path: '/api/auth/refresh-token',
        summary: 'Renovar token JWT',
        description: 'Genera un nuevo token JWT a partir del token actual',
        security: [['bearerAuth' => []]],
        tags: ['Autenticación'],
        responses: [
            new OA\Response(
                response: 200,
                description: 'Token renovado exitosamente',
                content: new OA\JsonContent(
                    properties: [
                        new OA\Property(property: 'token', type: 'string', example: 'eyJ0eXAiOiJKV1QiLCJhbGc...'),
                        new OA\Property(property: 'token_type', type: 'string', example: 'Bearer'),
                        new OA\Property(property: 'expires_in', type: 'integer', example: 3600),
                    ]
                )
            ),
            new OA\Response(
                response: 401,
                description: 'Token expirado o inválido',
                content: new OA\JsonContent(
                    properties: [
                        new OA\Property(property: 'message', type: 'string', example: 'Token expirado, ya no puede ser renovado'),
                    ]
                )
            ),
        ]
    )]
    public function refreshToken(Request $request)
    {

        try {
            $token = JWTAuth::getToken();
            $user = JWTAuth::setToken($token)->toUser();

            $newToken = JWTAuth::fromUser($user);

            //Sin recalcular los claims
            // $newToken = JWTAuth::parseToken()->refresh(false, true);

            return response()->json([
                'token' => $newToken,
                'token_type' => 'Bearer',
                'expires_in' => JWTAuth::factory()->getTTL() * 60
            ]);
        } catch (TokenExpiredException $e) {
            return response()->json(['message' => 'Token expirado, ya no puede ser renovado'], 401);
        } catch (JWTException $e) {
            return response()->json(['message' => 'Token inválido'], 401);
        }
    }
}
