<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Producto;
use Illuminate\Http\Request;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Symfony\Component\HttpFoundation\Response;

class InventariosController extends Controller
{
    private const NUMERIC_RULE = 'nullable|numeric|min:0';
    public function establecerStock(Request $request)
    {
        $items = $this->extraerItems($request);
        if ($items === null) {
            return response()->json(['message' => 'No se recibieron productos para actualizar'], Response::HTTP_BAD_REQUEST);
        }
        $errores = $this->validarItems($items);
        if ($errores) {
            return response()->json(['message' => 'Errores de validación', 'errors' => $errores], Response::HTTP_UNPROCESSABLE_ENTITY);
        }
        $actualizados = $this->actualizarProductos($items);
        return response()->json(['message' => 'Actualización completada', 'actualizados' => $actualizados]);
    }

    private function extraerItems(Request $request): ?array
    {
        $items = $request->input('productos');
        return (is_array($items) && !empty($items)) ? $items : null;
    }

    private function reglasItem(): array
    {
        return [
            'id' => 'required|integer|exists:cat_productos,idproduct',
            'stock_minimo' => self::NUMERIC_RULE,
            'costo_compra' => self::NUMERIC_RULE,
            'precio_venta' => self::NUMERIC_RULE,
            'activo' => 'nullable|boolean',
            'idfamilia' => 'nullable|integer',
            'idmarca' => 'nullable|integer'
        ];
    }

    private function validarItems(array $items): array
    {
        $errores = [];
        $reglas = $this->reglasItem();
        foreach ($items as $idx => $row) {
            $v = Validator::make($row, $reglas);
            if ($v->fails()) {
                $errores[$idx] = $v->errors()->all();
            }
        }
        return $errores;
    }

    private function actualizarProductos(array $items): int
    {
        $actualizados = 0;
        DB::transaction(function () use ($items, &$actualizados) {
            foreach ($items as $row) {
                $prod = Producto::find($row['id']);
                if (!$prod) {
                    continue;
                }
                $map = [
                    'stock_minimo' => 'stockminimo',
                    'costo_compra' => 'precioultimacompra',
                    'precio_venta' => 'precioventaconiva',
                    'activo' => 'productoactivo',
                    'idfamilia' => 'idfamilia',
                    'idmarca' => 'idmarca'
                ];
                $update = [];
                foreach ($map as $k => $col) {
                    if (array_key_exists($k, $row)) {
                        $val = $row[$k];
                        if ($k === 'activo') {
                            $val = $val ? 1 : 0;
                        }
                        $update[$col] = $val;
                    }
                }
                if ($update) {
                    $prod->fill($update)->save();
                    $actualizados++;
                }
            }
        });
        return $actualizados;
    }
}
