<?php

namespace App\Livewire\Actions;

use App\Enums\AlertEvent;
use App\Enums\AlertOption;
use App\Enums\AlertIcon;
use App\Traits\SweetAlert2;
use Exception;
use Livewire\Component;

class LivewireAlert
{

    use SweetAlert2;

    protected $component;

    protected array $options = [];
    /** @phpstan-ignore missingType.iterableValue */
    protected array $events = [];
    /** @phpstan-ignore missingType.iterableValue */

    public function __construct(Component $component) {
        $this->component = $component;
    } //  default implementation ignored

    private function isComponent()
    {
        return $this->component instanceof Component;
    }

    public function title(string $title): self
    {
        $this->options[AlertOption::Title->value] = $title;

        return $this;
    }

    public function text(string $text): self
    {
        $this->options[AlertOption::Text->value] = $text;

        return $this;
    }

    public function success(): self
    {
        $this->options[AlertOption::Icon->value] = AlertIcon::Success->value;

        return $this;
    }

    public function error(): self
    {
        $this->options[AlertOption::Icon->value] = AlertIcon::Error->value;

        return $this;
    }

    public function warning(): self
    {
        $this->options[AlertOption::Icon->value] =  AlertIcon::Warning->value;

        return $this;
    }

    public function info(): self
    {
        $this->options[AlertOption::Icon->value] = AlertIcon::Info->value;

        return $this;
    }

    public function question(): self
    {
        $this->options[AlertOption::Icon->value] = AlertIcon::Question->value;

        return $this;
    }

    public function timer(int $timer): self
    {
        $this->options[AlertOption::Timer->value] = $timer;

        return $this;
    }

    public function withConfirmButton(?string $confirmButtonText = null): self
    {
        $this->options[AlertOption::ShowConfirmButton->value] = true;

        $this->confirmButtonText(
            $confirmButtonText ?? config('livewire-alert.confirmButtonText')
        );

        return $this;
    }

    public function withCancelButton(?string $cancelButtonText = null): self
    {
        $this->options[AlertOption::ShowCancelButton->value] = true;

        $this->cancelButtonText(
            $cancelButtonText ?? config('livewire-alert.cancelButtonText')
        );

        return $this;
    }

    public function withDenyButton(?string $denyButtonText = null): self
    {
        $this->options[AlertOption::ShowDenyButton->value] = true;

        $this->denyButtonText(
            $denyButtonText ?? config('livewire-alert.denyButtonText')
        );

        return $this;
    }

    public function confirmButtonText(string $text): self
    {
        $this->options[AlertOption::ConfirmButtonText->value] = $text;

        return $this;
    }

    public function cancelButtonText(string $text): self
    {
        $this->options[AlertOption::CancelButtonText->value] = $text;

        return $this;
    }

    public function denyButtonText(string $text): self
    {
        $this->options[AlertOption::DenyButtonText->value] = $text;

        return $this;
    }

    public function asConfirm(): self
    {

        $this->withConfirmButton(config('livewire-alert.confirmButtonText'));
        $this->withDenyButton(config('livewire-alert.denyButtonText'));
        $this->options[AlertOption::Timer->value] = null;

        return $this;
    }

    public function onConfirm(string $action, mixed $data = null): self
    {
        $this->event(AlertEvent::IsConfirmed, [
            'action' => $action,
            'data' => $data,
        ]);

        return $this;
    }

    public function onDeny(string $action, mixed $data = null): self
    {
        $this->event(AlertEvent::IsDenied, [
            'action' => $action,
            'data' => $data,
        ]);

        return $this;
    }

    public function onDismiss(string $action, mixed $data = null): self
    {
        $this->event(AlertEvent::IsDismissed, [
            'action' => $action,
            'data' => $data,
        ]);

        return $this;
    }

    public function withOptions(array $options = []): self
    {
        $this->options = array_merge(
            $this->options,
            $options
        );

        return $this;
    }

    public function getOptions(): array
    {
        $defaultOptions = [
            'timer' => null,
            'text' => null,
            'confirmButtonText' => 'Yes',
            'cancelButtonText' => 'Cancel',
            'denyButtonText' => 'No',
            'showCancelButton' => false,
            'showConfirmButton' => false,
        ];

        return array_merge(
            $defaultOptions,
            array_intersect_key(
                $this->options,
                array_flip(AlertOption::values())
            ),
        );
    }

    public function getEvents(): array
    {
        return array_intersect_key(
            $this->events,
            array_flip(AlertEvent::values())
        );
    }

    public function show(): void
    {
        throw_if(
            !$this->isComponent(),
            new Exception('LivewireAlert neceitsa un componente para mostrarse')
        );

        $this->alert(
            $this->getOptions(),
            $this->getEvents()
        );
    }

    /**
     * @param array{'action': string, "data": array<array<string, mixed>>} $action
     */
    protected function event(AlertEvent $event, array $action): void
    {
        $this->events[$event->value] = $action;
    }
}
