<?php

namespace App\Livewire\Catalogos\Productos;

use App\Models\Familia;
use App\Models\Marca;
use App\Models\Producto;
use App\Traits\WithLivewireAlert;
use Illuminate\Support\Facades\Auth;
use Livewire\Attributes\Title;
use Livewire\Component;
use Livewire\WithPagination;
use Maatwebsite\Excel\Facades\Excel;
use App\Exports\Productos\ProductoExportExcel;

#[Title('Catalogo de Productos')]
class Index extends Component
{
    use WithPagination;
    use WithLivewireAlert;
    public $search = '';
    protected $queryString = ['search'];

    public $headerValues;
    public $families;
    public $brands;

    public $errorMessage;
    //Filters
    public $filters = [
        'family' => '',
        'brand' => ''
    ];
    // Esta funcion viene del ciclo de vida de los componentes de livewire
    public function mount()
    {
        $this->headerValues = [
            'Clave',
            'Código de barras',
            'Descripción',
            'Opciones'
        ];

        $this->families = Familia::all()->map(function ($family) {
            return ['idfamily' => $family->idfamily, 'nombrefamilia' => $family->nombrefamilia];
        })->prepend(['idfamily' => '', 'nombrefamilia' => 'Todos']);

        $this->brands = Marca::all()->map(function ($brand) {
            return ['marca' => $brand->marca, 'idbrand' => $brand->idbrand];
        })->prepend(['marca' => 'Todos', 'idbrand' => '']);
    }

    public function render()
    {
        $productos = Producto::searchOr(['clave', 'codigodebarras'], $this->search)
            ->search('idmarca', $this->filters['brand'])
            ->search('idfamilia', $this->filters['family'])
            ->paginate(10);
        return view('livewire.catalogos.productos.index', [
            'productos' => $productos
        ]);
    }

    public function updatingSearch()
    {
        $this->resetPage();
    }

    public function updatingFilters()
    {
        $this->resetPage();
    }

    public function clean() {
        $this->reset('filters', 'search');
    }

    public function create() {
        redirect()->route('d-catalogos-productos-create');
    }

    public function edit($productId) {
        redirect()->route('d-catalogos-productos-edit', ['producto' => $productId]);
    }

    public function delete(array $data) {
        $productId = $data['productId'];
        try {
            $producto = Producto::where('idproduct', $productId)
            ->firstOrFail();
            $producto->delete();

            $this->livewireAlert
            ->success()
            ->text($producto->nombrecorto . ' se ha eliminado correctamente')
            ->timer(1500)
            ->show();
            
        } catch (\Exception $e) {
            $this->livewireAlert
            ->error()
            ->text('No se ha encontrado el producto con id ' . strval($productId) . '.\n')
            ->timer(3000)
            ->show();
        }
    }

    public function confirmDelete(Producto $producto) {
        $this->livewireAlert
            ->warning()
            ->text('¿Estas seguro de eliminar ' . strval($producto->nombrecorto) . '?')
            ->withConfirmButton('Aceptar')
            ->withDenyButton('Cancelar')
            ->onConfirm('delete', ['productId' => $producto->idproduct])
            ->show();
    }
    
    public function exportExcel()
    {
        return Excel::download(
            new ProductoExportExcel($this->idempresa, $this->search, $this->filters),
            'productos.xlsx'
        );
    }

}
