<?php

namespace App\Livewire\Catalogos\Productos;

use App\Enums\BarcodeType;
use App\Enums\OrigenRegistro;
use App\Models\Producto;
use App\Traits\CurrencyValidator;
use App\Traits\ResetNested;
use Flux\Flux;
use Milon\Barcode\DNS1D;
use Illuminate\Support\Str;
use Illuminate\Validation\ValidationException;
use stdClass;

trait ProductBaseLogic
{
    use CurrencyValidator;
    use ResetNested;
    public $selectedUnidad;
    public $selectedProductService;
    public $selectedProductComponent;
    public $isClaveAvailable;
    public $removedProductComponents = [];
    public $showComponents = false;
    public $barcodeImage;
    public $codigo = '';

    public $search = [
        'unidades' => '',
        'productos_servicios' => '',
        'componentes' => '',
        'componente_criterio' => 'descripcion'
    ];
    public $activeTab = 'general';
    public $form = [
        'idfamilia' => '',
        'idmarca' => '',
        'clave' => '',
        'imagen' => null,
        'clave_unidadmedida' => '',
        'unidadmedida' => '',
        'clave_prodserv' => '',
        'prodserv' => '',
        'habilitar_pesaje' => false,
        'pesaje' => 0,
        'manejacaducidad' => false,
        'productoactivo' => true,
        'leerbascula' => false,
        'productoexclusivo' =>  false,
        'manejanumeroserie' => false,
        'manejainventario' => false,
        'esvisibleenventas' => true,
        'obtener_datos_from_cbb' => false,
        'aplicar_tipodecambio' => false,
        'esproductocompuesto' => false,
        'manejainsumos' => false,
        'nombrecorto' => '',
        'descripcion' => '',
        'precioultimacompra' => '0',
        'traslado_iva_tasa_cuota' => '0',
        'traslado_iva_exento' => false,
        'traslado_ieps_tasa_cuota' => '0',
        'fechaultimacompra' => '',
        'retencion_iva_tasa_cuota' => '0',
        'retencion_isr_tasa_cuota' => '0',
        'costo_promedio_antesiva' => '0',
        'costo_promedio_iva' => '0',
        'costo_promedio_despuesiva' => '0',
        'precioventaantesiva' => '0',
        'porcentajeutilidad' => '0',
        'precioventaconiva' => '0',
        'nota1' => '',
        'nota2' => '',
        'nota3' => '',
        'productoactivo' => true,
        'fechaalta' => '',
        'stockminimo' => '0',
        'comision1x100' => '0',
        'comision2x100' => '0',
        'comision3x100' => '0',
        'codigodebarras' => '',
        'idempresa' => null,
        'idproducto' => -1,
        'comisiontipo' => 0,

    ];
    public $precios = [
        'ultimacompra_con_imp' => '0',
        'ultimacompra_iva' => '0',
        'ultimacompra_ieps' => '0',
        'venta_iva_traslado' => '0',
        'utilidad_costo_promedio' => '0',
        'venta_ieps_traslado' => '0',
        'costo_promedio_ieps' => '0'
    ];

    public $cambiarCostosPromedio = false;

    protected $rules = [
        'form.clave' => 'required|max:30',
        'form.nombrecorto' => 'required|min:2|max:30',
        'form.descripcion' => 'required|min:2|max:255',
        'form.idmarca' => 'required|max:30',
        'form.idfamilia' => 'required|max:30',
        'form.precioultimacompra' => 'required|decimal:0,6|gt:0',
        'form.precioventaantesiva' => 'required|decimal:0,6|gt:0',
        'form.clave_unidadmedida' => 'required|max:10',
        'form.unidadmedida' => 'required|max:150',
        'form.clave_prodserv' => 'required|max:10',
        'form.prodserv' => 'required|max:150',
        'form.habilitar_pesaje' => 'required',
        'form.pesaje' => 'required',
        'form.manejacaducidad' => 'required',
        'form.productoactivo' => 'required',
        'form.leerbascula' => 'required',
        'form.productoexclusivo' => 'required',
        'form.manejanumeroserie' => 'required',
        'form.manejainventario' => 'required',
        'form.esvisibleenventas' => 'required',
        'form.obtener_datos_from_cbb' => 'required',
        'form.aplicar_tipodecambio' => 'required',
        'form.esproductocompuesto' => 'required',
        'form.manejainsumos' => 'required',
        'form.traslado_iva_tasa_cuota' => 'required|decimal:0,6|gte:0',
        'form.traslado_iva_exento' => 'required',
        'form.traslado_ieps_tasa_cuota' => 'required|decimal:0,6|gte:0',
        'form.retencion_iva_tasa_cuota' => 'required|decimal:0,6|gte:0',
        'form.retencion_isr_tasa_cuota' => 'required|decimal:0,6|gte:0',
        'form.costo_promedio_antesiva' => 'required|decimal:0,6|gt:0',
        'form.costo_promedio_iva' => 'required|decimal:0,6|gte:0',
        'form.costo_promedio_despuesiva' => 'required|decimal:0,6|gt:0',
        'form.precioventaantesiva' => 'required|decimal:0,6|gt:0',
        'form.porcentajeutilidad' => 'required|decimal:0,6',
        'form.precioventaconiva' => 'required|decimal:0,6|gt:0',
        'form.nota1' => 'max:400',
        'form.nota2' => 'max:400',
        'form.nota3' => 'max:400',
    ];
    protected $atributtes = [
        'form.clave' => 'clave',
        'form.nombrecorto' => 'nombre corto',
        'form.precioultimacompra' => 'precio ultima compra',
        'form.precioventaantesiva' => 'precio venta sin impuestos',
        'form.habilitar_pesaje' => 'habilitar pesaje',
        'form.pesaje' => 'pesaje',
        'form.manejacaducidad' => 'maneja caducidad',
        'form.productoactivo' => 'producto activo',
        'form.leerbascula' => 'leer bascula',
        'form.productoexclusivo' => 'producto exclusivo',
        'form.manejanumeroserie' => 'maneja numero de serie',
        'form.manejainventario' => 'maneja inventario',
        'form.esvisibleenventas' => 'es visible en ventas',
        'form.obtener_datos_from_cbb' => 'obtner datos de cbb',
        'form.aplicar_tipodecambio' => 'aplicar tipo de cambio',
        'form.esproductocompuesto' => 'es producto compuesto',
        'form.manejainsumos' => 'maneja insumos',
        'form.traslado_iva_tipo_factor' => 'traslado iva factor',
        'form.traslado_iva_tasa_cuota' => 'traslado iva cuota',
        'form.traslado_iva_exento' => 'traslado iva exento',
        'form.traslado_ieps_tipo_factor' => 'traslado ieps factor',
        'form.traslado_ieps_tasa_cuota' => 'traslado ieps cuota',
        'form.retencion_iva_tipo_factor' => 'retencion iva factor',
        'form.retencion_iva_tasa_cuota' => 'retencion iva cuota',
        'form.retencion_isr_tipo_factor' => 'retencion isr factor',
        'form.retencion_isr_tasa_cuota' => 'retencion isr cuota',
        'form.costo_promedio_antesiva' => 'costo promedio sin impuestos',
        'form.costo_promedio_iva' => 'costo promedio iva',
        'form.costo_promedio_despuesiva' => 'costo promedio con impuestos',
        'form.porcentajeutilidad' => 'porcentaje de utilidad',
        'form.precioventaconiva' => 'precio venta con impuestos',
        'form.descripcion' => 'descripcion',
        'form.clave_unidadmedida' => 'clave unidad',
        'form.unidadmedida' => 'unidad de medida',
    ];


    public function updatingSearch()
    {
        $target = &$this;
        $target->resetPage();
    }

    public function getRandomCode()
    {
        $this->codigo = Str::random(10);
    }

    public function generateBarcode()
    {
        if ($this->codigo == '') {
            return;
        }

        $barcode = new DNS1D();
        $barcode->setStorPath(storage_path('app/public/barcodes'));

        $this->barcodeImage = $barcode->getBarcodePNG(
            code: $this->codigo,
            type: BarcodeType::Alphanumeric->value,
            w: 1.5,
            h: 27,
            showCode: true
        );
        Flux::modal('codigo-barras')->close();
        $this->form['codigodebarras'] = $this->codigo;
    }

    public function resetModal()
    {

        $this->reset(['search', 'selectedUnidad']);
    }

    public function saveUnidad()
    {
        $this->form['clave_unidadmedida'] = $this->selectedUnidad['clave_unidad'];
        $this->form['unidadmedida'] = $this->selectedUnidad['nombre'];
        Flux::modal('unidad-medida')->close();
    }

    public function saveProductService()
    {
        $this->form['clave_prodserv'] = $this->selectedProductService['clave_prodserv'];
        $this->form['prodserv'] = $this->selectedProductService['descripcion'] ?? $this->selectedProductService['nombre'];
        Flux::modal('producto-servicio')->close();
    }

    public function clear()
    {
        $this->reset(['form', 'precios']);
    }

    public function calcularTrasladoDesdePrecioBase()
    {
        if (!$this->isValidCurrency($this->form['precioultimacompra'])) {
            $this->resetNestedMultiple(
                value: '0',
                propertyPaths: ['precios.ultimacompra_con_imp', 'form.traslado_ieps_tasa_cuota', 'form.traslado_iva_tasa_cuota', 'form.costo_promedio_antesiva', 'form.costo_promedio_iva', 'form.costo_promedio_despuesiva']
            );
            return;
        }

        $precio = $this->form['precioultimacompra'];
        $ieps = $this->calcularTasa($precio, $this->form['traslado_ieps_tasa_cuota']);
        $iva = $this->calcularTasa($precio, $this->form['traslado_iva_tasa_cuota']);
        $total = bcadd(bcadd($precio, $iva, 6), $ieps, 6);

        $this->precios['ultimacompra_ieps'] = $ieps;
        $this->precios['ultimacompra_iva'] = $iva;
        $this->precios['ultimacompra_con_imp'] = $total;

        $this->form['costo_promedio_antesiva'] = $precio;
        $this->precios['costo_promedio_ieps'] = $ieps;
        $this->form['costo_promedio_iva'] = $iva;
        $this->form['costo_promedio_despuesiva'] = $total;
    }

    public function calcularTrasladoDesdePrecioTotal()
    {
        if (!$this->isValidCurrency($this->precios['ultimacompra_con_imp'])) {
            $this->resetNestedMultiple(
                value: '0',
                propertyPaths: ['form.precioultimacompra', 'precios.ultimacompra_iva', 'precios.ultimacompra_ieps', 'form.costo_promedio_antesiva', 'form.costo_promedio_iva', 'form.costo_promedio_despuesiva']
            );
            return;
        }

        $total = $this->precios['ultimacompra_con_imp'];
        $factor = bcadd(
            bcadd(
                $this->form['traslado_ieps_tasa_cuota'],
                $this->form['traslado_iva_tasa_cuota'],
                6
            ),
            '1',
            6
        );

        $precio = bcdiv($total, $factor, 6);

        $ieps = $this->calcularTasa($precio, $this->form['traslado_ieps_tasa_cuota']);
        $iva = $this->calcularTasa($precio, $this->form['traslado_iva_tasa_cuota']);

        $this->precios['ultimacompra_ieps'] = $ieps;
        $this->precios['ultimacompra_iva'] = $iva;
        $this->form['precioultimacompra'] = $precio;

        $this->form['costo_promedio_antesiva'] = $precio;
        $this->precios['costo_promedio_ieps'] = $ieps;
        $this->form['costo_promedio_iva'] = $iva;
        $this->form['costo_promedio_despuesiva'] = $total;
    }

    public function calcularPreciosVentaDesdeBase()
    {
        if (!$this->isValidCurrency($this->form['precioventaantesiva'])) {
            $this->resetNestedMultiple(
                value: '0',
                propertyPaths: ['precios.venta_ieps_traslado', 'precios.venta_iva', 'form.precioventaconiva', 'form.porcentajeutilidad', 'precios.utilidad_costo_promedio']
            );
            return;
        }

        $precioVenta = $this->form['precioventaantesiva'];
        $ieps = $this->calcularTasa($precioVenta, $this->form['traslado_ieps_tasa_cuota']);
        $iva = $this->calcularTasa($precioVenta, $this->form['traslado_iva_tasa_cuota']);

        $total = bcadd(bcadd($precioVenta, $iva, 6), $ieps, 6);

        $this->precios['venta_iva_traslado'] = $iva;
        $this->precios['venta_ieps_traslado'] = $ieps;
        $this->form['precioventaconiva'] = $total;
    }

    public function calcularPreciosVentaDesdeTotal()
    {
        if (!$this->isValidCurrency($this->form['precioventaconiva'])) {
            $this->resetNestedMultiple(
                value: '0',
                propertyPaths: ['precios.venta_ieps_traslado', 'precios.venta_iva', 'precios.precioventaantesiva', 'form.porcentajeutilidad', 'precios.utilidad_costo_promedio']
            );
            return;
        }

        $total = $this->form['precioventaconiva'];
        $factor = bcadd(
            bcadd(
                $this->form['traslado_iva_tasa_cuota'],
                $this->form['traslado_ieps_tasa_cuota'],
                6
            ),
            '1',
            6
        );

        $precio = bcdiv($total, $factor, 6);

        $ieps = $this->calcularTasa($precio, $this->form['traslado_ieps_tasa_cuota']);
        $iva = $this->calcularTasa($precio, $this->form['traslado_iva_tasa_cuota']);

        $this->precios['venta_iva_traslado'] = $iva;
        $this->precios['venta_ieps_traslado'] = $ieps;
        $this->form['precioventaantesiva'] = $precio;
    }

    public function calcularVentasUtilidades()
    {
        if (!$this->isValidCurrency($this->form['precioultimacompra'])) {
            $this->resetNested('form.porcentajeutilidad', '0');
        } else {
            $this->form['porcentajeutilidad'] = bcmul(
                bcdiv(
                    bcsub(
                        $this->form['precioventaantesiva'],
                        $this->form['precioultimacompra'],
                        6
                    ),
                    $this->form['precioultimacompra'],
                    6
                ),
                '100',
                6
            );
        }

        if (!$this->isValidCurrency($this->form['costo_promedio_antesiva'])) {
            $this->resetNested('precios.utilidad_costo_promedio', '0');
            return;
        }

        $this->precios['utilidad_costo_promedio'] = bcmul(
            bcdiv(
                bcsub(
                    $this->form['precioventaantesiva'],
                    $this->form['costo_promedio_antesiva'],
                    6
                ),
                $this->form['costo_promedio_antesiva'],
                6
            ),
            '100',
            6
        );
    }
    private function validateClaveValue($value)
    {
        if ($value === '') {
            $this->resetNested('form.clave', '');
            $this->resetErrorBag('form.clave');
            return;
        }

        $foundProduct = Producto::whereRaw('LOWER(clave) = ?', [strtolower($value)])->first();
        $isSameProduct = $foundProduct && isset($this->form['idproduct']) && $foundProduct->idproduct === $this->form['idproduct'];
        if (!$foundProduct || $isSameProduct) {
            $this->isClaveAvailable = true;
            $this->resetErrorBag('form.clave');
            return;
        }
        $this->isClaveAvailable = false;
        throw ValidationException::withMessages([
            'form.clave' => ['La clave ya está en uso.'],
        ]);
    }

    public function calcularTasa($base, $factor): string
    {
        if (!$base || !$factor) {
            return '0.000000';
        }
        return bcmul($base, $factor, 6);
    }

    public function removeProductComponent(int $index)
    {
        $removedProductComponent = new stdClass;
        $removedProductComponent->originalIndex = $index;
        $removedProductComponent->value = $this->productComponents[$index];

        unset($this->productComponents[$index]);
        array_push($this->removedProductComponents, $removedProductComponent);
    }
    public function saveSelectectedComponent()
    {
        if (!isset($this->selectedProductComponent)) {
            return;
        }

        $newProductComponent = [];
        $newProductComponent['idcomponente'] = $this->selectedProductComponent["idproduct"];
        $newProductComponent['cantidad'] = 1;
        $newProductComponent['producto'] = $this->selectedProductComponent;
        array_push($this->productComponents, $newProductComponent);
        Flux::modal("buscar-producto-componentes")->close();
    }

    public function getPartialImageUrl()
    {
        return '/imagenes/productos/';
    }

    public function updatedForm($value, $key)
    {
        if ($key === 'clave') {
            $this->validateClaveValue($value);
            return;
        }

        if ($key === 'precioultimacompra' || $key === 'traslado_ieps_tasa_cuota' || $key === 'traslado_iva_tasa_cuota') {
            $this->calcularTrasladoDesdePrecioBase();
            $this->calcularPreciosVentaDesdeBase();
            $this->calcularVentasUtilidades();
        }

        if ($key === 'traslado_iva_exento' && $value) {
            $this->form['traslado_iva_tasa_cuota'] = 0;
            $this->calcularTrasladoDesdePrecioBase();
            $this->calcularPreciosVentaDesdeBase();
            $this->calcularVentasUtilidades();
        }

        if ($key === 'precioventaantesiva') {
            $this->calcularPreciosVentaDesdeBase();
            $this->calcularVentasUtilidades();
        }

        if ($key === 'precioventaconiva') {
            $this->calcularPreciosVentaDesdeTotal();
            $this->calcularVentasUtilidades();
        }

        if ($key === 'costo_promedio_antesiva') {
            $this->calcularVentasUtilidades();
        }
    }

    public function updatedPrecios($value, $key)
    {
        if ($key === 'ultimacompra_con_imp') {
            $this->calcularTrasladoDesdePrecioTotal();
            $this->calcularPreciosVentaDesdeBase();
            $this->calcularVentasUtilidades();
        }
    }

    protected function saveProductImage($image, $idproduct) {
        $productImageName = 'prod_' . $idproduct . '.' . $image->guessExtension();
        $image->storeAs(
            path: $this->getPartialImageUrl(),
            name: $productImageName
        );
        return $this->getPartialImageUrl() . $productImageName;
    }
}
