<?php

namespace App\Livewire\Catalogos\Usuario;

use Livewire\Component;
use Livewire\Attributes\Title;
use Livewire\Attributes\On;
use App\Models\Usuario;
use App\Models\Perfil;
use App\Models\Sucursal;
use App\Models\Permiso;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

#[Title('Creando Usuario')]
class Create extends Component
{
    public $activeTab = 'general';
    public $idempresa = '';
    public $nombre = '';
    public $usuario = '';
    public $pasword = '';
    public $confirmpasword = '';
    public $perfil = '';
    public $perfiles = [];
    public $sucursal = '';
    public $sucursales = [];
    public $permisos = [];
    public $alert_title = 'Usuario creado';
    public $alert_text = 'Se creó el usuario correctamente';
    

    protected $rules = [
        'nombre' => 'required|string|max:255',
        'usuario' => 'required|string|max:255',
        'pasword' => 'required|string|max:255',
        'confirmpasword' => 'required|string|same:pasword',
        'perfil'              => 'required|integer|exists:perfiles,idperfil',
        'sucursal'            => 'required|integer|exists:cat_sucursales,idsucursal',
    ];

    public function mount()
    {
        $menus = DB::table('menus')->get();
        $menuopciones = DB::table('menuopciones')->get()->groupBy('idmenu');
        foreach ($menus as $menu) {
            if (isset($menuopciones[$menu->idmenu])) {
                $this->permisos[$menu->idmenu] = $menuopciones[$menu->idmenu]->map(function ($opcion) {
                    return [
                        'idmenuopcion' => $opcion->idmenuopcion,
                        'nombre' => $opcion->descripcion,
                        'permitir' => false,
                    ];
                })->toArray();
            } else {
                $this->permisos[$menu->idmenu] = [];
            }
        }

        $this->perfiles = Perfil::orderBy('nivel_acceso')
            ->pluck('nombre', 'idperfil')
            ->toArray();
        $this->sucursales = Sucursal::where('idempresa', Auth::user()->idempresa)
            ->orderBy('razon_social')
            ->pluck('razon_social', 'idsucursal')
            ->toArray();
    }

    public function render()
    {
        return view('livewire.catalogos.usuario.form', [
            'perfiles' => $this->perfiles,
            'sucursales' => $this->sucursales,
        ]);
    }

    public function save()
    {
        $this->validate();

        DB::transaction(function () {
            $usuario = Usuario::create([
                'idusuario'          => 0,
                'idempresa'          => Auth::user()->idempresa,
                'nombre'             => $this->nombre,
                'usuario'            => $this->usuario,
                'pasword'            => $this->pasword,
                'idperfil'           => $this->perfil,
                'idsucursal'         => $this->sucursal,
                'nopermitirborrarreg' => in_array($this->perfil, [1001, 2001]) ? 0 : 1,
                'origen_registro'    => 1,
            ]);

            foreach ($this->permisos as $idmenu => $options) {
                foreach ($options as $option) {
                    $permiso = Permiso::create([
                        'idempresa'      => Auth::user()->idempresa,
                        'idsucursal'     => $this->sucursal,
                        'iddesk'         => 1,
                        'idusuario'      => $usuario->iduser,
                        'idperfil'       => $this->perfil,
                        'idmenu'         => $idmenu,
                        'idmenuopcion'   => $option['idmenuopcion'],
                        'permitir'       => $option['permitir'] ? 1 : 0,
                        'origen_registro' => 1,
                    ]);
                }
            }
        });
        $this->dispatch('create-alert');
    }

    #[On('on-accept')]
    public function onAccept()
    {
        return redirect()->route('d-catalogos-usuarios');
    }

    public function cancel()
    {
        return $this->dispatch('go-back');
    }

    public function clear()
    {
        $this->reset();
    }
}
