<?php

namespace App\Livewire\Inventarios;

use App\Enums\ActionOption;
use App\Models\Almacen;
use App\Models\Concepto;
use App\Models\Producto;
use App\Models\Proveedor;
use App\Repositories\Inventarios\InventariosRepositoryInterface;
use App\Support\StringUtils;
use App\Traits\WithLivewireAlert;
use Flux\Flux;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\ValidationException;
use Livewire\Attributes\Title;
use Livewire\Component;
use Livewire\WithPagination;

#[Title('Inventarios - Movimientos')]
class Movimiento extends Component
{
    use WithPagination;
    use WithLivewireAlert;
    use \App\Traits\WithFormRequest;

    public $search = [
        'sucursal' => '',
        'proveedor' => '',
        'producto' => '',
    ];

    public $conceptos = [];
    public $warehouses = [];
    public $selectedSucursal = null;
    public $searchWarehouseId = null;

    public $selectedConceptoId = null;

    public $selectedProducto = null;
    public $searchProductoId = null;

    public $selectedProveedor = null;
    public $searchProveedorId = null;

    public $partidasActuales = [];
    public $selectedPartida = null;
    public $partidaForm = [
        'cantidad' => '',
        'precio' => '',
        'importe' => '',
        'lote' => '',
        'fecha_caducidad' => '',
        'justificacion' => '',
    ];

    public $importeTotal = '';
    public $fechaMovimiento = null;
    public string $movOption = 'entrada';

    protected $rules = [
        'selectedSucursal.idwarehouse' => 'required|exists:cat_almacenes,idwarehouse',
        'selectedProveedor.idprovider' => 'nullable|exists:cat_proveedores,idprovider',
        'selectedConceptoId' => 'required|exists:cat_conceptosinventarios,idinventoryconcept',
        'partidasActuales' => 'array|min:1|max:100',
        'partidasActuales.*.cantidad' => 'required|numeric|min:1',
        'partidasActuales.*.lote' => 'string|max:50',
        'partidasActuales.*.fecha_caducidad' => 'required|date|after:yesterday',
        'partidasActuales.*.justificacion' => 'string|max:255',
        'importeTotal' => 'required|numeric|min:0',
        'fechaMovimiento' => 'required|date',
        'movOption' => 'required|in:entrada,salida',
    ];

    public function mount()
    {
        $conceptos = Concepto::getConceptosByIdAccion(ActionOption::ENTRADA->value);
        $this->conceptos = $conceptos;
        $this->warehouses = Almacen::with('sucursal')->get();

        $defaultConcepto = $conceptos->first()->idinventoryconcept ?? null;
        $this->selectedConceptoId = $defaultConcepto;
        $this->fechaMovimiento = now()->format('Y-m-d');
    }

    public function render()
    {
        $productos = [];
        if (!StringUtils::isEmpty($this->search['producto']) && isset($this->selectedSucursal->idwarehouse)) {
            $productos = Producto::searchOr(['nombrecorto', 'clave', 'codigodebarras'], $this->search['producto'])
                ->whereHas('inventarios', function ($query) {
                    $query->where('idalmacen', $this->selectedSucursal->idwarehouse);
                })
                ->paginate(10);
        }

        $proveedores = [];
        if (!StringUtils::isEmpty($this->search['proveedor'])) {
            $proveedores = Proveedor::search('nombre', $this->search['proveedor'])->paginate(10);
        }

        return view('livewire.inventarios.movimientos.index', [
            'productos' => $productos,
            'proveedores' => $proveedores,
        ]);
    }

    public function resetModal()
    {

        $this->reset(['search']);
    }

    public function updatingSearch()
    {
        $this->resetPage();
    }

    public function saveSelectedSucursal()
    {
        Flux::modal('sucursal-modal')->close();
        $warehouse = Almacen::with('sucursal')->find($this->searchWarehouseId);
        $this->selectedSucursal = $warehouse;
    }

    public function saveSelectedProducto()
    {
        Flux::modal('producto-modal')->close();
        $producto = DB::table('cat_productos as p')
            ->where('p.idproduct', $this->searchProductoId)
            ->join('mov_inv_existencias as i', 'i.idproducto', '=', 'p.idproduct')
            ->where('i.idalmacen', $this->selectedSucursal->idwarehouse)
            ->select('p.*', 'i.existencia')
            ->first();

        $this->selectedProducto = $producto;
        $this->partidaForm['cantidad'] = '1';
        $this->partidaForm['importe'] = bcmul($producto->precioultimacompra, $this->partidaForm['cantidad'], 6);
        $this->partidaForm['precio'] = $producto->precioultimacompra;
        $this->partidaForm['lote'] = '';
        $this->partidaForm['fecha_caducidad'] = now()->format('Y-m-d');
    }

    public function saveSelectedProveedor()
    {
        Flux::modal('proveedor-modal')->close();
        $proveedor = Proveedor::find($this->searchProveedorId);
        $this->selectedProveedor = $proveedor;
    }

    public function updatedPartidaFormCantidad($value)
    {
        if (is_numeric($value) && $value < 0) {
            $this->partidaForm['cantidad'] = 0;
            $value = 0;
        }
        $this->partidaForm['importe'] = bcmul($this->partidaForm['precio'], $value, 6);
    }

    public function updatedMovOption($value)
    {
        $accion = ActionOption::fromLabel($value);
        $conceptos = Concepto::getConceptosByIdAccion($accion->value);
        $this->conceptos = $conceptos;
        $this->selectedConceptoId = $conceptos->first()->idinventoryconcept ?? null;
    }

    public function addPartida()
    {
        if (!$this->isValidPartida()) {
            return;
        }
        $this->partidasActuales[] = [
            'cantidad' => $this->partidaForm['cantidad'],
            'precio' => $this->partidaForm['precio'],
            'importe' => $this->partidaForm['importe'],
            'lote' => $this->partidaForm['lote'],
            'justificacion' => $this->partidaForm['justificacion'],
            'fecha_caducidad' => $this->partidaForm['fecha_caducidad'],
            'producto' => $this->selectedProducto,
            'producto_id' => $this->selectedProducto->idproduct,
        ];
        $this->reset('partidaForm');
        $this->selectedProducto = null;
        $this->importeTotal = $this->calcularImporteTotal();
    }

    public function removePartida($index)
    {
        if (!isset($index) && !isset($this->partidasActuales[$index])) {
            return;
        }
        unset($this->partidasActuales[$index]);
        $this->partidasActuales = array_values($this->partidasActuales);
        $this->importeTotal = $this->calcularImporteTotal();
    }

    private function calcularImporteTotal()
    {
        return array_reduce($this->partidasActuales, fn($total, $partida) =>
        bcadd($total, $partida['importe'] ?? 0, 6), 0);
    }

    private function isValidPartida(): bool
    {
        return isset($this->selectedProducto) && $this->validate([
            'partidaForm.cantidad' => 'required|numeric|min:1',
            'partidaForm.lote' => 'string|max:50',
            'partidaForm.fecha_caducidad' => 'required|date|after:yesterday',
            'partidaForm.justificacion' => 'string|max:255',
        ]);
    }

    public function clearPartidas()
    {
        $this->partidasActuales = [];
        $this->importeTotal = '';
        $this->selectedSucursal = null;
        $this->selectedProducto = null;
        $this->selectedProveedor = null;
    }

    public function confirmClearPartidas()
    {
        $this->livewireAlert
            ->text('¿Estás seguro de que deseas limpiar todas las partidas?')
            ->withConfirmButton('Limpiar')
            ->withCancelButton('Cancelar')
            ->onConfirm('clearPartidas')
            ->show();
    }

    private function getFormData(): array
    {
        return [
            'partidas' => $this->partidasActuales,
            'almacen_id' => $this->selectedSucursal['idwarehouse'] ?? null,
            'concepto_id' => $this->selectedConceptoId,
            'importe_total' => $this->importeTotal,
            'tipo_movimiento' => $this->movOption,
            'proveedor_id' => $this->selectedProveedor,
            'fecha_movimiento' => $this->fechaMovimiento,
            'importe_total' => $this->importeTotal,
        ];
    }

    public function finalizePartidas()
    {
        DB::beginTransaction();
        try {
            $repository = app(InventariosRepositoryInterface::class);
            $repository->registrar(
                $this->getFormData()
            );
        } catch (ValidationException $e) {
            DB::rollBack();
            if (isset($e->errors()['partidas'])) {
                $this->livewireAlert
                    ->text('Es necesario agregar al menos una partida para continuar.')
                    ->withConfirmButton('Aceptar')
                    ->show();
                return;
            }

            $this->livewireAlert
                ->title('Errores en el formulario')
                ->text(collect($e->errors())->flatten()->first() ?: 'Por favor verifica los datos ingresados.')
                ->withConfirmButton('Aceptar')
                ->show();
            throw $e;
        } catch (\Exception $e) {
            DB::rollBack();
            $this->livewireAlert
                ->text('Ocurrió un error inesperado. Por favor intenta nuevamente.')
                ->withConfirmButton('Aceptar')
                ->show();
            return;
        }

        DB::commit();
        $this->livewireAlert
            ->text('El movimiento de inventario se ha registrado correctamente.')
            ->withConfirmButton('Aceptar')
            ->show();
        $this->clearPartidas();
    }
}
