<?php

namespace App\Livewire\Inventarios;

use App\Enums\ActionOption;
use App\Models\Almacen;
use App\Models\Concepto;
use App\Models\Producto;
use App\Models\Proveedor;
use App\Repositories\Inventarios\InventariosRepositoryInterface;
use App\Support\StringUtils;
use App\Traits\WithLivewireAlert;
use Flux\Flux;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Validation\ValidationException;
use Livewire\Attributes\Title;
use Livewire\Component;
use Livewire\WithPagination;

#[Title('Inventarios - Traspasos')]
class Traspaso extends Component
{
    use WithPagination;
    use WithLivewireAlert;
    use \App\Traits\WithFormRequest;

    public $search = [
        'sucursalOrigen' => '',
        'sucursalDestino' => '',
        'proveedor' => '',
        'producto' => '',
    ];

    public $conceptos = [];
    public $warehousesOrigen = [];
    public $warehousesDestino = [];
    
    public $selectedSucursalOrigen = null;
    public $searchWarehouseOrigenId = null;
    
    public $selectedSucursalDestino = null;
    public $searchWarehouseDestinoId = null;

    public $selectedConceptoId = null;

    public $selectedProducto = null;
    public $searchProductoId = null;

    public $selectedProveedor = null;
    public $searchProveedorId = null;

    public $partidasActuales = [];
    public $selectedPartida = null;
    public $partidaForm = [
        'cantidad' => '',
        'precio' => '',
        'importe' => '',
        'lote_origen' => '',
        'fecha_caducidad' => '',
        'justificacion' => '',
    ];

    public $importeTotal = '';
    public $fechaTraspaso = null;

    protected $rules = [
        'selectedSucursalOrigen.idwarehouse' => 'required|exists:cat_almacenes,idwarehouse',
        'selectedSucursalDestino.idwarehouse' => 'required|exists:cat_almacenes,idwarehouse|different:selectedSucursalOrigen.idwarehouse',
        'selectedProveedor.idprovider' => 'nullable|exists:cat_proveedores,idprovider',
        'selectedConceptoId' => 'required|exists:cat_conceptosinventarios,idinventoryconcept',
        'partidasActuales' => 'array|min:1|max:300',
        'partidasActuales.*.cantidad' => 'required|numeric|min:1',
        'partidasActuales.*.lote_origen' => 'string|max:50',
        'partidasActuales.*.fecha_caducidad' => 'nullable|date|after:yesterday',
        'partidasActuales.*.justificacion' => 'string|max:255',
        'importeTotal' => 'required|numeric|min:0',
        'fechaTraspaso' => 'required|date',
    ];

    public function mount()
    {
        // Obtener conceptos de traspaso
        $conceptos = Concepto::getConceptosByIdAccion(ActionOption::TRASPASO->value);
        
        // Si no hay conceptos de traspaso, usar los de entrada como fallback
        if ($conceptos->isEmpty()) {
            $conceptos = Concepto::getConceptosByIdAccion(ActionOption::ENTRADA->value);
        }
        
        $this->conceptos = $conceptos;
        
        $almacenes = Almacen::with('sucursal')->get();
        $this->warehousesOrigen = $almacenes;
        $this->warehousesDestino = $almacenes;

        $defaultConcepto = $conceptos->first()->idinventoryconcept ?? null;
        $this->selectedConceptoId = $defaultConcepto;
        $this->fechaTraspaso = now()->format('Y-m-d');
    }

    public function render()
    {
        $productos = [];
        if (!StringUtils::isEmpty($this->search['producto']) && isset($this->selectedSucursalOrigen->idwarehouse)) {
            $productos = Producto::searchOr(['nombrecorto', 'clave', 'codigodebarras'], $this->search['producto'])
                ->whereHas('inventarios', function ($query) {
                    $query->where('idalmacen', $this->selectedSucursalOrigen->idwarehouse);
                })
                ->paginate(10);
        }

        $proveedores = [];
        if (!StringUtils::isEmpty($this->search['proveedor'])) {
            $proveedores = Proveedor::search('nombre', $this->search['proveedor'])->paginate(10);
        }

        return view('livewire.inventarios.traspasos.index', [
            'productos' => $productos,
            'proveedores' => $proveedores,
        ]);
    }

    public function resetModal()
    {
        $this->reset(['search']);
    }

    public function updatingSearch()
    {
        $this->resetPage();
    }

    public function saveSelectedSucursalOrigen()
    {
        Flux::modal('sucursal-origen-modal')->close();
        $warehouse = Almacen::with('sucursal')->find($this->searchWarehouseOrigenId);
        $this->selectedSucursalOrigen = $warehouse;
    }

    public function saveSelectedSucursalDestino()
    {
        Flux::modal('sucursal-destino-modal')->close();
        
        // Validar que no sea la misma sucursal
        if ($this->selectedSucursalOrigen && 
            $this->searchWarehouseDestinoId == $this->selectedSucursalOrigen->idwarehouse) {
            $this->livewireAlert
                ->text('La sucursal destino debe ser diferente a la sucursal origen.')
                ->withConfirmButton('Aceptar')
                ->show();
            return;
        }
        
        $warehouse = Almacen::with('sucursal')->find($this->searchWarehouseDestinoId);
        $this->selectedSucursalDestino = $warehouse;
    }

    public function saveSelectedProducto()
    {
        Flux::modal('producto-modal')->close();
        $producto = DB::table('cat_productos as p')
            ->where('p.idproduct', $this->searchProductoId)
            ->join('mov_inv_existencias as i', 'i.idproducto', '=', 'p.idproduct')
            ->where('i.idalmacen', $this->selectedSucursalOrigen->idwarehouse)
            ->select('p.*', 'i.existencia')
            ->first();

        $this->selectedProducto = $producto;
        $this->partidaForm['cantidad'] = '1';
        $this->partidaForm['importe'] = bcmul($producto->precioultimacompra ?? 0, $this->partidaForm['cantidad'], 6);
        $this->partidaForm['precio'] = $producto->precioultimacompra ?? 0;
        $this->partidaForm['lote_origen'] = '';
        $this->partidaForm['fecha_caducidad'] = now()->format('Y-m-d');
    }

    public function saveSelectedProveedor()
    {
        Flux::modal('proveedor-modal')->close();
        $proveedor = Proveedor::find($this->searchProveedorId);
        $this->selectedProveedor = $proveedor;
    }

    public function updatedPartidaFormCantidad($value)
    {
        if (is_numeric($value) && $value < 0) {
            $this->partidaForm['cantidad'] = 0;
            $value = 0;
        }
        
        // Validar que no exceda la existencia
        if ($this->selectedProducto && $value > $this->selectedProducto->existencia) {
            $this->livewireAlert
                ->text('La cantidad no puede ser mayor a la existencia disponible (' . $this->selectedProducto->existencia . ').')
                ->withConfirmButton('Aceptar')
                ->show();
            $this->partidaForm['cantidad'] = $this->selectedProducto->existencia;
            $value = $this->selectedProducto->existencia;
        }
        
        $this->partidaForm['importe'] = bcmul($this->partidaForm['precio'], $value, 6);
    }

    public function addPartida()
    {
        if (!$this->isValidPartida()) {
            return;
        }
        
        // Validar que las sucursales estén seleccionadas
        if (!$this->selectedSucursalOrigen || !$this->selectedSucursalDestino) {
            $this->livewireAlert
                ->text('Debe seleccionar tanto la sucursal origen como la sucursal destino.')
                ->withConfirmButton('Aceptar')
                ->show();
            return;
        }
        
        $this->partidasActuales[] = [
            'cantidad' => $this->partidaForm['cantidad'],
            'precio' => $this->partidaForm['precio'],
            'importe' => $this->partidaForm['importe'],
            'lote_origen' => $this->partidaForm['lote_origen'],
            'justificacion' => $this->partidaForm['justificacion'],
            'fecha_caducidad' => $this->partidaForm['fecha_caducidad'],
            'producto' => $this->selectedProducto,
            'producto_id' => $this->selectedProducto->idproduct,
        ];
        $this->reset('partidaForm');
        $this->selectedProducto = null;
        $this->importeTotal = $this->calcularImporteTotal();
    }

    public function removePartida($index)
    {
        if (!isset($index) && !isset($this->partidasActuales[$index])) {
            return;
        }
        unset($this->partidasActuales[$index]);
        $this->partidasActuales = array_values($this->partidasActuales);
        $this->importeTotal = $this->calcularImporteTotal();
    }

    private function calcularImporteTotal()
    {
        return array_reduce($this->partidasActuales, fn($total, $partida) =>
        bcadd($total, $partida['importe'] ?? 0, 6), 0);
    }

    private function isValidPartida(): bool
    {
        return isset($this->selectedProducto) && $this->validate([
            'partidaForm.cantidad' => 'required|numeric|min:1',
            'partidaForm.lote_origen' => 'string|max:50',
            'partidaForm.fecha_caducidad' => 'nullable|date|after:yesterday',
            'partidaForm.justificacion' => 'string|max:255',
        ]);
    }

    public function clearPartidas()
    {
        $this->partidasActuales = [];
        $this->importeTotal = '';
        $this->selectedSucursalOrigen = null;
        $this->selectedSucursalDestino = null;
        $this->selectedProducto = null;
        $this->selectedProveedor = null;
    }

    public function confirmClearPartidas()
    {
        $this->livewireAlert
            ->text('¿Estás seguro de que deseas limpiar todas las partidas?')
            ->withConfirmButton('Limpiar')
            ->withCancelButton('Cancelar')
            ->onConfirm('clearPartidas')
            ->show();
    }

    private function getFormData(): array
    {
        return [
            'partidas' => $this->partidasActuales,
            'almacen_origen_id' => $this->selectedSucursalOrigen['idwarehouse'] ?? null,
            'almacen_destino_id' => $this->selectedSucursalDestino['idwarehouse'] ?? null,
            'concepto_id' => $this->selectedConceptoId,
            'importe_total' => $this->importeTotal,
            'tipo_movimiento' => 'traspaso',
            'proveedor_id' => $this->selectedProveedor->idprovider ?? null,
            'fecha_movimiento' => $this->fechaTraspaso,
        ];
    }

    public function finalizePartidas()
    {
        // Validar que las sucursales sean diferentes
        if ($this->selectedSucursalOrigen && $this->selectedSucursalDestino &&
            $this->selectedSucursalOrigen->idwarehouse == $this->selectedSucursalDestino->idwarehouse) {
            $this->livewireAlert
                ->text('El almacén de destino debe ser diferente al almacén de origen.')
                ->withConfirmButton('Aceptar')
                ->show();
            return;
        }
        
        Log::info('Iniciando transacción de traspaso');
        DB::beginTransaction();
        try {
            $repository = app(InventariosRepositoryInterface::class);
            
            Log::info('Llamando a registrarTraspaso');
            // Usar el nuevo método específico para traspasos
            $resultado = $repository->registrarTraspaso(
                $this->getFormData()
            );
            
            Log::info('registrarTraspaso completado, haciendo commit', ['resultado' => $resultado]);
            DB::commit();
            Log::info('Transacción commiteada exitosamente');
            
            // Mostrar mensaje de éxito con información de ambos movimientos
            $this->livewireAlert
                ->title('Traspaso Exitoso')
                ->text($resultado['message'] ?? 'El traspaso se ha registrado correctamente.')
                ->withConfirmButton('Aceptar')
                ->show();
                
            $this->clearPartidas();
            
        } catch (ValidationException $e) {
            Log::warning('ValidationException en traspaso', ['errors' => $e->errors()]);
            DB::rollBack();
            Log::info('Transacción revertida por ValidationException');
            
            if (isset($e->errors()['partidas'])) {
                $this->livewireAlert
                    ->text('Es necesario agregar al menos una partida para continuar.')
                    ->withConfirmButton('Aceptar')
                    ->show();
                return;
            }

            $this->livewireAlert
                ->title('Errores en el formulario')
                ->text(collect($e->errors())->flatten()->first() ?: 'Por favor verifica los datos ingresados.')
                ->withConfirmButton('Aceptar')
                ->show();
                
        } catch (\Exception $e) {
            Log::error('Exception en traspaso', [
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            DB::rollBack();
            Log::info('Transacción revertida por Exception');
            $this->livewireAlert
                ->text('Ocurrió un error inesperado: ' . $e->getMessage())
                ->withConfirmButton('Aceptar')
                ->show();
        }
    }
}
