<?php

namespace App\Livewire\Reportes\CortesCaja;

use App\Exports\CortesCajaExport;
use App\Models\CortesTurnos;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Livewire\Attributes\Title;
use Livewire\Component;
use Livewire\WithPagination;
use Maatwebsite\Excel\Facades\Excel;
use Barryvdh\DomPDF\Facade\Pdf;

#[Title('Cortes de Caja')]
class Index extends Component
{
    use WithPagination;

    public $fechaInicio;
    public $fechaFin;
    public $idSucursal = null;
    public $idUsuario = null;
    public $opcionTurnos = 2; // 0=No cerrados, 1=Cerrados, 2=Ambos
    public $search = '';
    
    public $idEmpresa;
    public $sucursales = [];
    public $usuarios = [];
    
    // Modal de filtro de turnos
    public $mostrarModalTurnos = false;
    public $tipoFiltroTurnos = 'no_considerar'; // no_considerar, rango, precisos
    public $turnoDesde = '';
    public $turnoHasta = '';
    public $turnosPrecisos = ''; // Ejemplo: 23,45,67
    public $considerarFechas = false;

    protected $queryString = [
        'fechaInicio',
        'fechaFin',
        'idSucursal',
        'idUsuario',
        'opcionTurnos',
        'search',
        'tipoFiltroTurnos',
        'turnoDesde',
        'turnoHasta',
        'turnosPrecisos'
    ];

    public function mount()
    {
        $this->idEmpresa = Auth::user()->tenant_id;
        $this->fechaInicio = $this->fechaInicio ?? date('Y-m-d');
        $this->fechaFin = $this->fechaFin ?? date('Y-m-d');
        
        $this->cargarSucursales();
        $this->cargarUsuarios();
    }

    public function cargarSucursales()
    {
        $this->sucursales = DB::table('cat_sucursales')
            ->select('idsucursal', 'razon_social')
            ->get();
    }

    public function cargarUsuarios()
    {
        $query = DB::table('cat_usuarios')
            ->select('idusuario', 'nombre', 'idsucursal');
        
        if ($this->idSucursal) {
            $query->where('idsucursal', $this->idSucursal);
        }
        
        $this->usuarios = $query->get();
    }

    public function updatedIdSucursal()
    {
        $this->cargarUsuarios();
        $this->idUsuario = null;
        $this->resetPage();
    }

    public function updatingSearch()
    {
        $this->resetPage();
    }

    public function render()
    {
        $cortes = $this->obtenerCortes();

        $totales = $this->calcularTotales($cortes);

        return view('livewire.reportes.cortes-caja.index', [
            'cortes' => $cortes,
            'totales' => $totales
        ]);
    }

    private function obtenerCortes()
    {
        $query = DB::table('cortes_turnos as c')
            ->select(
                'c.idturn',
                'c.idsucursal',
                'c.idturno',
                'c.fechahora_apertura',
                'c.turno_cerrado',
                'c.fechahora_cierre',
                'c.importe_caja_inicial',
                'c.total_corte',
                'u.nombre as usuario',
                'j.nombre_caja',
                's.razon_social as sucursal',
                DB::raw('COALESCE(ing.ingresos, 0) as ingresos'),
                DB::raw('COALESCE(eng.egresos, 0) as egresos')
            )
            ->leftJoin('cat_usuarios as u', function ($join) {
                $join->on('u.idusuario', '=', 'c.idusuario')
                    ->on('u.idsucursal', '=', 'c.idsucursal');
            })
            ->leftJoin('cajas as j', function ($join) {
                $join->on('j.idcaja', '=', 'c.idcaja_apertura')
                    ->on('j.idsucursal', '=', 'c.idsucursal');
            })
            ->leftJoin('cat_sucursales as s', 's.idsucursal', '=', 'c.idsucursal')
            ->leftJoin(DB::raw("(
                SELECT idturno, SUM(importe) as ingresos
                FROM cortes_turnos_detalle
                WHERE tipo = 2
                GROUP BY idturno
            ) as ing"), 'ing.idturno', '=', 'c.idturn')
            ->leftJoin(DB::raw("(
                SELECT idturno, SUM(importe) as egresos
                FROM cortes_turnos_detalle
                WHERE tipo = 3
                GROUP BY idturno
            ) as eng"), 'eng.idturno', '=', 'c.idturn');

        // Filtros
        if ($this->fechaInicio) {
            $query->where('c.fechahora_apertura', '>=', $this->fechaInicio . ' 00:00:00');
        }

        if ($this->fechaFin) {
            $query->where('c.fechahora_apertura', '<=', $this->fechaFin . ' 23:59:59');
        }

        if ($this->idSucursal) {
            $query->where('c.idsucursal', $this->idSucursal);
        }

        if ($this->idUsuario) {
            $query->where('c.idusuario', $this->idUsuario);
        }

        // Filtro por estado del turno
        if ($this->opcionTurnos == 0) {
            $query->where('c.turno_cerrado', 0);
        } elseif ($this->opcionTurnos == 1) {
            $query->where('c.turno_cerrado', 1);
        }

        // Búsqueda
        if ($this->search) {
            $query->where(function ($q) {
                $q->where('c.idturno', 'like', '%' . $this->search . '%')
                    ->orWhere('u.nombre', 'like', '%' . $this->search . '%')
                    ->orWhere('j.nombre_caja', 'like', '%' . $this->search . '%');
            });
        }
        
        // Filtro avanzado de turnos
        $this->aplicarFiltroTurnos($query);

        return $query->orderBy('c.fechahora_apertura', 'desc')
            ->paginate(10);
    }
    
    private function aplicarFiltroTurnos($query)
    {
        if ($this->tipoFiltroTurnos === 'rango' && $this->turnoDesde && $this->turnoHasta) {
            $query->whereBetween('c.idturno', [(int)$this->turnoDesde, (int)$this->turnoHasta]);
        } elseif ($this->tipoFiltroTurnos === 'precisos' && $this->turnosPrecisos) {
            $turnos = array_map('trim', explode(',', $this->turnosPrecisos));
            $turnos = array_filter($turnos, 'is_numeric');
            if (!empty($turnos)) {
                $query->whereIn('c.idturno', $turnos);
            }
        }
        // 'no_considerar' no aplica filtros adicionales de turnos
    }

    private function calcularTotales($cortes)
    {
        $totalCortes = 0;
        $totalIngresos = 0;
        $totalEgresos = 0;

        foreach ($cortes as $corte) {
            $totalCortes += $corte->total_corte;
            $totalIngresos += $corte->ingresos;
            $totalEgresos += $corte->egresos;
        }

        return [
            'total_cortes' => $totalCortes,
            'total_ingresos' => $totalIngresos,
            'total_egresos' => $totalEgresos,
            'diferencia' => ($totalIngresos - $totalEgresos) - $totalCortes
        ];
    }

    public function verDetalle($idTurno, $idSucursal)
    {
        return redirect()->route('d-reportes-cortes-caja-detalle', [
            'idturno' => $idTurno,
            'idsucursal' => $idSucursal
        ]);
    }

    public function limpiarFiltros()
    {
        $this->fechaInicio = date('Y-m-d');
        $this->fechaFin = date('Y-m-d');
        $this->idSucursal = null;
        $this->idUsuario = null;
        $this->opcionTurnos = 2;
        $this->search = '';
        $this->tipoFiltroTurnos = 'no_considerar';
        $this->turnoDesde = '';
        $this->turnoHasta = '';
        $this->turnosPrecisos = '';
        $this->considerarFechas = false;
        $this->resetPage();
    }
    
    public function abrirModalTurnos()
    {
        $this->mostrarModalTurnos = true;
    }
    
    public function cerrarModalTurnos()
    {
        $this->mostrarModalTurnos = false;
    }
    
    public function aplicarFiltroTurnosModal()
    {
        $this->mostrarModalTurnos = false;
        $this->resetPage();
    }
    
    public function consultar()
    {
        // Ejecuta la consulta normal (ya se hace automáticamente con Livewire)
        // Solo forzamos un refresh
        $this->resetPage();
    }
    
    public function consultarConcentrado()
    {
        // Genera un reporte concentrado (resumen general sin detalle por turno)
        return redirect()->route('d-reportes-cortes-caja-concentrado', [
            'fechaInicio' => $this->fechaInicio,
            'fechaFin' => $this->fechaFin,
            'idSucursal' => $this->idSucursal,
            'idUsuario' => $this->idUsuario,
            'opcionTurnos' => $this->opcionTurnos
        ]);
    }
    
    public function exportarExcel()
    {
        $cortes = $this->obtenerCortesSinPaginacion();
        $totales = $this->calcularTotales(collect($cortes));
        
        $sucursalNombre = null;
        if ($this->idSucursal) {
            foreach ($this->sucursales as $suc) {
                if ($suc->idsucursal == $this->idSucursal) {
                    $sucursalNombre = $suc->razon_social;
                    break;
                }
            }
        }
        
        $usuarioNombre = null;
        if ($this->idUsuario) {
            foreach ($this->usuarios as $usr) {
                if ($usr->idusuario == $this->idUsuario) {
                    $usuarioNombre = $usr->nombre;
                    break;
                }
            }
        }
        
        $filtros = [
            'fechaInicio' => $this->fechaInicio,
            'fechaFin' => $this->fechaFin,
            'sucursal' => $sucursalNombre,
            'usuario' => $usuarioNombre,
            'opcionTurnos' => $this->opcionTurnos,
        ];
        
        $filename = 'cortes_caja_' . date('Y-m-d_His') . '.xlsx';
        
        return Excel::download(new CortesCajaExport($cortes, $totales, $filtros), $filename);
    }
    
    public function exportarPDF()
    {
        $cortes = $this->obtenerCortesSinPaginacion();
        $totales = $this->calcularTotales(collect($cortes));
        
        $sucursalNombre = null;
        if ($this->idSucursal) {
            foreach ($this->sucursales as $suc) {
                if ($suc->idsucursal == $this->idSucursal) {
                    $sucursalNombre = $suc->razon_social;
                    break;
                }
            }
        }
        
        $usuarioNombre = null;
        if ($this->idUsuario) {
            foreach ($this->usuarios as $usr) {
                if ($usr->idusuario == $this->idUsuario) {
                    $usuarioNombre = $usr->nombre;
                    break;
                }
            }
        }
        
        $filtros = [
            'fechaInicio' => $this->fechaInicio,
            'fechaFin' => $this->fechaFin,
            'sucursal' => $sucursalNombre,
            'usuario' => $usuarioNombre,
            'opcionTurnos' => $this->opcionTurnos,
        ];
        
        $pdf = Pdf::loadView('pdf.cortes-caja', [
            'cortes' => $cortes,
            'totales' => $totales,
            'filtros' => $filtros,
        ]);
        
        $pdf->setPaper('letter', 'landscape');
        
        $filename = 'cortes_caja_' . date('Y-m-d_His') . '.pdf';
        
        return response()->streamDownload(function() use ($pdf) {
            echo $pdf->stream();
        }, $filename);
    }
    
    private function obtenerCortesSinPaginacion()
    {
        // Copia la misma lógica de obtenerCortes() pero sin paginar
        $query = DB::table('cortes_turnos as c')
            ->select(
                'c.idturn',
                'c.idsucursal',
                'c.idturno',
                'c.fechahora_apertura',
                'c.turno_cerrado',
                'c.fechahora_cierre',
                'c.importe_caja_inicial',
                'c.total_corte',
                'u.nombre as usuario',
                'j.nombre_caja',
                's.razon_social as sucursal',
                DB::raw('COALESCE(ing.ingresos, 0) as ingresos'),
                DB::raw('COALESCE(eng.egresos, 0) as egresos')
            )
            ->leftJoin('cat_usuarios as u', function ($join) {
                $join->on('u.idusuario', '=', 'c.idusuario')
                    ->on('u.idsucursal', '=', 'c.idsucursal');
            })
            ->leftJoin('cajas as j', function ($join) {
                $join->on('j.idcaja', '=', 'c.idcaja_apertura')
                    ->on('j.idsucursal', '=', 'c.idsucursal');
            })
            ->leftJoin('cat_sucursales as s', 's.idsucursal', '=', 'c.idsucursal')
            ->leftJoin(DB::raw("(
                SELECT idturno, SUM(importe) as ingresos
                FROM cortes_turnos_detalle
                WHERE tipo = 2
                GROUP BY idturno
            ) as ing"), 'ing.idturno', '=', 'c.idturn')
            ->leftJoin(DB::raw("(
                SELECT idturno, SUM(importe) as egresos
                FROM cortes_turnos_detalle
                WHERE tipo = 3
                GROUP BY idturno
            ) as eng"), 'eng.idturno', '=', 'c.idturn');

        // Aplicar los mismos filtros
        if ($this->fechaInicio) {
            $query->where('c.fechahora_apertura', '>=', $this->fechaInicio . ' 00:00:00');
        }

        if ($this->fechaFin) {
            $query->where('c.fechahora_apertura', '<=', $this->fechaFin . ' 23:59:59');
        }

        if ($this->idSucursal) {
            $query->where('c.idsucursal', $this->idSucursal);
        }

        if ($this->idUsuario) {
            $query->where('c.idusuario', $this->idUsuario);
        }

        if ($this->opcionTurnos == 0) {
            $query->where('c.turno_cerrado', 0);
        } elseif ($this->opcionTurnos == 1) {
            $query->where('c.turno_cerrado', 1);
        }

        if ($this->search) {
            $query->where(function ($q) {
                $q->where('c.idturno', 'like', '%' . $this->search . '%')
                    ->orWhere('u.nombre', 'like', '%' . $this->search . '%')
                    ->orWhere('j.nombre_caja', 'like', '%' . $this->search . '%');
            });
        }
        
        $this->aplicarFiltroTurnos($query);

        return $query->orderBy('c.fechahora_apertura', 'desc')->get();
    }
}
