<?php

namespace App\Livewire\Reportes;

use Livewire\Component;
use App\Models\Sucursal;
use App\Models\Producto;
use App\Models\Familia;
use Livewire\WithPagination;
use Livewire\Attributes\Title;
use Maatwebsite\Excel\Facades\Excel;
use App\Exports\Reportes\ExistenciasPorSucursalesExport;
use Mpdf\Mpdf;

#[Title('Existencias por Sucursales')]
class VentasPorSucursales extends Component
{
    use WithPagination;

    protected $paginationTheme = 'tailwind';

    public $isOpenModalExportarReporte = false;
    public $search = '';
    public $familia = '';
    public $criterioExistencias = 'todos';
    public $considerarProductosInactivos = false;
    public $tipoBusqueda = '';
    
    public $sucursales = [];
    public $familias = [];

    protected $queryString = [
        'search' => ['except' => ''],
        'familia' => ['except' => ''],
        'criterioExistencias' => ['except' => 'todos'],
        'considerarProductosInactivos' => ['except' => false],
        'tipoBusqueda' => ['except' => '']
    ];

    public function mount()
    {
        $this->cargarSucursales();
        $this->cargarFamilias();
    }

    public function cargarSucursales()
    {
        $this->sucursales = Sucursal::select('idbranchoffice', 'razon_social')
            ->orderBy('idbranchoffice')
            ->get();
    }

    public function cargarFamilias()
    {
        $this->familias = Familia::select('idfamily', 'nombrefamilia')
            ->orderBy('clave')
            ->get();
    }

    public function updatingSearch()
    {
        $this->resetPage();
    }

    public function updatingFamilia()
    {
        $this->resetPage();
    }

    public function updatingCriterioExistencias()
    {
        $this->resetPage();
    }

    public function updatingTipoBusqueda()
    {
        $this->resetPage();
    }

    public function obtenerProductos()
    {
        // Query base usando Eloquent con relaciones
        $query = Producto::query()
            ->leftJoin('mov_inv_existencias as mie', function($join) {
                $join->on('mie.idproducto', '=', 'cat_productos.idproduct');
            })
            ->leftJoin('cat_familias as f', function($join) {
                $join->on('f.idfamily', '=', 'cat_productos.idfamilia');
            })
            ->join('cat_sucursales as s', 's.idbranchoffice', '=', 'mie.idsucursal')
            ->select(
                'cat_productos.idproduct',
                'cat_productos.clave',
                //'cat_productos.folio_correlativo',
                'cat_productos.descripcion',
                'mie.existencia',
                'cat_productos.stockminimo',
                'f.nombrefamilia as familia',
                's.razon_social as sucursal',
                's.idbranchoffice'
            )
            ->where('cat_productos.manejainventario', 1);

        // Filtro de búsqueda
        if ($this->search) {
            $query->where(function($q) {
                $q->where('cat_productos.descripcion', 'like', '%' . $this->search . '%')
                  //->orWhere('cat_productos.folio_correlativo', 'like', '%' . $this->search . '%')
                  ->orWhere('cat_productos.clave', 'like', '%' . $this->search . '%')
                  ->orWhere('f.nombrefamilia', 'like', '%' . $this->search . '%');
            });
        }

        // Filtro de familia
        if ($this->familia) {
            $query->where('cat_productos.idfamilia', $this->familia);
        }

        // Filtro de tipo de búsqueda (productos sobrantes, stock mínimo, etc.)
        if ($this->tipoBusqueda === 'productossobrantes') {
            $query->whereColumn('mie.existencia', '>', 'cat_productos.stockminimo');
        } elseif ($this->tipoBusqueda === 'stockminimo_igual_a_existencia') {
            $query->whereColumn('mie.existencia', '=', 'cat_productos.stockminimo');
        } elseif ($this->tipoBusqueda === 'stockminimo0_y_existencia0') {
            $query->where('mie.existencia', 0)
                  ->where('cat_productos.stockminimo', 0);
        }

        // Filtro de criterio de existencias
        if ($this->criterioExistencias != 'todos') {
            $query->where('mie.existencia', $this->criterioExistencias, 0);
        }

        // Filtro de productos activos/inactivos
        if ($this->considerarProductosInactivos) {
            $query->whereIn('cat_productos.productoactivo', [0, 1]);
        } else {
            $query->where('cat_productos.productoactivo', 1);
        }

        $productos = $query
            ->orderBy('cat_productos.idproduct')
            ->orderBy('s.idbranchoffice')
            ->paginate(50);

        // Agrupar productos por id para mostrar existencias por sucursal
        $productosAgrupados = $productos->getCollection()->groupBy('idproduct');
        
        $productosFinales = collect();
        foreach ($productosAgrupados as $items) {
            $primerItem = $items->first();
            $producto = (object)[
                'idproduct' => $primerItem->idproduct,
                'folio_correlativo' => $primerItem->folio_correlativo,
                'clave' => $primerItem->clave,
                'descripcion' => $primerItem->descripcion,
                'familia' => $primerItem->familia,
                'existencias' => []
            ];
            
            // Construir array de existencias por sucursal
            foreach ($items as $item) {
                $producto->existencias[$item->idbranchoffice] = $item->existencia ?? 0;
            }
            
            // Completar sucursales faltantes con 0
            foreach ($this->sucursales as $sucursal) {
                if (!isset($producto->existencias[$sucursal->idbranchoffice])) {
                    $producto->existencias[$sucursal->idbranchoffice] = 0;
                }
            }
            
            $productosFinales->push($producto);
        }

        //Mantener el objeto paginador intacto
        return new \Illuminate\Pagination\LengthAwarePaginator(
            $productosFinales,
            $productos->total(),
            $productos->perPage(),
            $productos->currentPage(),
            ['path' => request()->url(), 'query' => request()->query()]
        );
    }

    public function exportarPdfReporteVentasPorSucursales()
    {
        // Validar que haya datos
        if ($this->obtenerProductos()->isEmpty()) {
            session()->flash('error', 'No hay datos para exportar. Por favor, ajusta los filtros.');
            $this->isOpenModalExportarReporte = false;
            return;
        }

        // Obtener todos los productos sin paginación
        $productos = $this->obtenerProductosSinPaginacion();
        
        // Obtener nombre de familia si hay filtro
        $familiaSeleccionada = null;
        if ($this->familia) {
            $familiaObj = $this->familias->firstWhere('idfamily', $this->familia);
            $familiaSeleccionada = $familiaObj ? $familiaObj->nombrefamilia : null;
        }

        $html = view('exports.existencias-por-sucursales', [
            'productos' => $productos,
            'sucursales' => $this->sucursales,
            'search' => $this->search,
            'familia' => $this->familia,
            'familiaSeleccionada' => $familiaSeleccionada,
            'tipoBusqueda' => $this->tipoBusqueda,
        ])->render();

        $mpdf = new Mpdf([
            'mode' => 'utf-8',
            'format' => 'A4-L', // Formato horizontal para más columnas
            'margin_left' => 10,
            'margin_right' => 10,
            'margin_top' => 10,
            'margin_bottom' => 20,
        ]);

        $mpdf->WriteHTML($html);

        $filename = 'existencias-por-sucursales-' . date('Ymd_His') . '.pdf';

        return response()->streamDownload(function () use ($mpdf) {
            $mpdf->Output('', 'D');
        }, $filename);
    }

    public function exportarExcelReporteVentasPorSucursales()
    {
        // Validar que haya datos
        if ($this->obtenerProductos()->isEmpty()) {
            session()->flash('error', 'No hay datos para exportar. Por favor, ajusta los filtros.');
            $this->isOpenModalExportarReporte = false;
            return;
        }

        // Obtener todos los productos sin paginación
        $productos = $this->obtenerProductosSinPaginacion();

        $filename = 'existencias-por-sucursales-' . date('Ymd_His') . '.xlsx';

        return Excel::download(
            new ExistenciasPorSucursalesExport($productos, $this->sucursales),
            $filename
        );
    }

    /**
     * Obtiene productos sin paginación para exportación
     */
    private function obtenerProductosSinPaginacion()
    {
        // Query base usando Eloquent con relaciones
        $query = Producto::query()
            ->leftJoin('mov_inv_existencias as mie', function($join) {
                $join->on('mie.idproducto', '=', 'cat_productos.idproduct');
            })
            ->leftJoin('cat_familias as f', function($join) {
                $join->on('f.idfamily', '=', 'cat_productos.idfamilia');
            })
            ->join('cat_sucursales as s', 's.idbranchoffice', '=', 'mie.idsucursal')
            ->select(
                'cat_productos.idproduct',
                'cat_productos.clave',
                //'cat_productos.folio_correlativo',
                'cat_productos.descripcion',
                'mie.existencia',
                'cat_productos.stockminimo',
                'f.nombrefamilia as familia',
                's.razon_social as sucursal',
                's.idbranchoffice'
            )
            ->where('cat_productos.manejainventario', 1);

        if ($this->search) {
            $query->where(function($q) {
                $q->where('cat_productos.descripcion', 'like', '%' . $this->search . '%')
                  ->orWhere('cat_productos.folio_correlativo', 'like', '%' . $this->search . '%')
                  ->orWhere('cat_productos.clave', 'like', '%' . $this->search . '%')
                  ->orWhere('f.nombrefamilia', 'like', '%' . $this->search . '%');
            });
        }

        if ($this->familia) {
            $query->where('cat_productos.idfamilia', $this->familia);
        }

        if ($this->tipoBusqueda === 'productossobrantes') {
            $query->whereColumn('mie.existencia', '>', 'cat_productos.stockminimo');
        } elseif ($this->tipoBusqueda === 'stockminimo_igual_a_existencia') {
            $query->whereColumn('mie.existencia', '=', 'cat_productos.stockminimo');
        } elseif ($this->tipoBusqueda === 'stockminimo0_y_existencia0') {
            $query->where('mie.existencia', 0)
                  ->where('cat_productos.stockminimo', 0);
        }

        if ($this->criterioExistencias != 'todos') {
            $query->where('mie.existencia', $this->criterioExistencias, 0);
        }

        if ($this->considerarProductosInactivos) {
            $query->whereIn('cat_productos.productoactivo', [0, 1]);
        } else {
            $query->where('cat_productos.productoactivo', 1);
        }

        $resultados = $query
            ->orderBy('cat_productos.idproduct')
            ->orderBy('s.idbranchoffice')
            ->get();

        // Agrupar productos por id para mostrar existencias por sucursal
        $productosAgrupados = $resultados->groupBy('idproduct');
        
        $productosFinales = collect();
        foreach ($productosAgrupados as $items) {
            $primerItem = $items->first();
            $producto = (object)[
                'idproduct' => $primerItem->idproduct,
                'folio_correlativo' => $primerItem->folio_correlativo,
                'clave' => $primerItem->clave,
                'descripcion' => $primerItem->descripcion,
                'familia' => $primerItem->familia,
                'existencias' => []
            ];
            
            // Construir array de existencias por sucursal
            foreach ($items as $item) {
                $producto->existencias[$item->idbranchoffice] = $item->existencia ?? 0;
            }
            
            // Completar sucursales faltantes con 0
            foreach ($this->sucursales as $sucursal) {
                if (!isset($producto->existencias[$sucursal->idbranchoffice])) {
                    $producto->existencias[$sucursal->idbranchoffice] = 0;
                }
            }
            
            $productosFinales->push($producto);
        }

        return $productosFinales;
    }
    
    public function render()
    {
        $productos = $this->obtenerProductos();
        
        return view('livewire.reportes.ventas-por-sucursales.index', [
            'productos' => $productos
        ]);
    }
}
