<?php

namespace App\Livewire\Reportes\Ventaspor;

use Livewire\Attributes\Title;
use Livewire\Component;
use App\Models\Sucursal;
use App\Models\Usuario;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

#[Title('Ventas por Facturación')]
class Index extends Component
{
    private const FECHA_CFDI_RAW = 'DATE(c.fechahora)';
    private const FECHA_VENTA_RAW = 'DATE(v.fechaventa)';
    private const CFDI_TABLE = 'cfdi_general01 as c';
    private const SUCURSAL_TABLE = 'cat_sucursales as s';
    private const VENTAS_TABLE = 'ventasgeneral as v';
    public $fechainicio;
    public $fechafinal;
    public $sucursal = '';
    public $usuario = '';
    public $cliente = '';
    public $turno = '';
    public $tipopago = '';
    public $origenVenta = 'todos';
    public $estatus = 'cerrada';
    public $filas = 10;

    public function mount()
    {
        $this->fechainicio = now()->startOfMonth()->format('Y-m-d');
        $this->fechafinal  = now()->endOfMonth()->format('Y-m-d');
    }
    protected function aplicarFiltrosFacturas($query)
    {
        if ($this->fechainicio && $this->fechafinal) {
            $query->whereBetween(DB::raw(self::FECHA_CFDI_RAW), [$this->fechainicio, $this->fechafinal]);
        }

        if ($this->sucursal) {
            $query->where('c.idsucursal', $this->sucursal);
        }

        if ($this->usuario) {
            $query->where('c.usuario', 'like', '%' . $this->usuario . '%');
        }

        if ($this->cliente) {
            $query->where('c.receptoridcliente', 'like', '%' . $this->cliente . '%');
        }

        if ($this->turno) {
            $query->where('c.idturno', $this->turno);
        }

        if ($this->tipopago) {
            $query->where('c.formadepago', $this->tipopago);
        }

        if ($this->estatus === 'cerrada') {
            $query->where('c.cancelada', 0);
        } elseif ($this->estatus === 'cancelada') {
            $query->where('c.cancelada', 1);
        }

        return $query;
    }

    protected function aplicarFiltrosVentas($query)
    {
        if ($this->fechainicio && $this->fechafinal) {
            $query->whereBetween(DB::raw(self::FECHA_VENTA_RAW), [$this->fechainicio, $this->fechafinal]);
        }

        if ($this->sucursal) {
            $query->where('v.idsucursal', $this->sucursal);
        }

        if ($this->usuario) {
            $query->where('v.usuario', 'like', '%' . $this->usuario . '%');
        }

        if ($this->cliente) {
            $query->where('v.idcliente', 'like', '%' . $this->cliente . '%');
        }

        if ($this->turno) {
            $query->where('v.idturno', $this->turno);
        }

        if ($this->tipopago) {
            $query->where('v.tipopago', $this->tipopago);
        }

        if ($this->estatus === 'cerrada') {
            $query->where('v.idstatus', 1);
        } elseif ($this->estatus === 'cancelada') {
            $query->where('v.idstatus', '!=', 1);
        }

        return $query;
    }

    public function render()
    {
        Log::info('Render llamado en Ventasporfacturacion', [
            'fechainicio' => $this->fechainicio,
            'fechafinal' => $this->fechafinal,
            'sucursal' => $this->sucursal,
            'estatus' => $this->estatus,
        ]);

        $queryFacturas = DB::table(self::CFDI_TABLE)
            ->leftJoin(self::SUCURSAL_TABLE, 's.idbranchoffice', '=', 'c.idsucursal')
            ->selectRaw("'factura' as origen, c.idweb as id, DATE(c.fechahora) as fecha, c.fechahora as fechahora, c.formadepago as tipopago, c.total as total, c.folio as folio, c.cfdi_uuid as foliofiscal, c.receptornombre as cliente, COALESCE(c.usuario, '') as usuario, COALESCE(s.razon_social, 'Sin sucursal') as sucursal, c.idsucursal, c.idturno, IF(c.cancelada = 0, 'cerrada', 'cancelada') as estatus");

        $this->aplicarFiltrosFacturas($queryFacturas);

        $queryVentasFacturas = DB::table(self::VENTAS_TABLE)
            ->leftJoin(self::SUCURSAL_TABLE, 's.idbranchoffice', '=', 'v.idsucursal')
            ->whereIn('v.idtipodocumento', [2, 10, 11, 12])
            ->selectRaw("'factura' as origen, v.idventageneral as id, " . self::FECHA_VENTA_RAW . " as fecha, v.fechaventa as fechahora, v.tipopago as tipopago, v.total as total, v.folio as folio, NULL as foliofiscal, v.cliente as cliente, COALESCE(v.usuario, '') as usuario, COALESCE(s.razon_social, 'Sin sucursal') as sucursal, v.idsucursal, v.idturno, IF(v.idstatus = 1, 'cerrada', 'cancelada') as estatus");

        $queryVentasNotas = DB::table(self::VENTAS_TABLE)
            ->leftJoin(self::SUCURSAL_TABLE, 's.idbranchoffice', '=', 'v.idsucursal')
            ->whereIn('v.idtipodocumento', [3, 7, 8, 9, 13, 14])
            ->selectRaw("'venta' as origen, v.idventageneral as id, " . self::FECHA_VENTA_RAW . " as fecha, v.fechaventa as fechahora, v.tipopago as tipopago, v.total as total, v.folio as folio, NULL as foliofiscal, v.cliente as cliente, COALESCE(v.usuario, '') as usuario, COALESCE(s.razon_social, 'Sin sucursal') as sucursal, v.idsucursal, v.idturno, IF(v.idstatus = 1, 'cerrada', 'cancelada') as estatus");

        $this->aplicarFiltrosVentas($queryVentasFacturas);
        $this->aplicarFiltrosVentas($queryVentasNotas);

        $facturasCfdiList = collect();
        $facturasVentasGeneralList = collect();
        $ventasNotasList = collect();

        if ($this->origenVenta === 'todos' || $this->origenVenta === 'factura') {
            $facturasCfdiList = $queryFacturas->orderBy('fechahora', 'desc')->limit($this->filas)->get();
            $facturasVentasGeneralList = $queryVentasFacturas->orderBy('fechahora', 'desc')->limit($this->filas)->get();
        }

        if ($this->origenVenta === 'todos' || $this->origenVenta === 'venta') {
            $ventasNotasList = $queryVentasNotas->orderBy('fechahora', 'desc')->limit($this->filas)->get();
        }

        $ventas = $facturasCfdiList
            ->concat($facturasVentasGeneralList)
            ->concat($ventasNotasList)
            ->sortByDesc('fechahora')
            ->values()
            ->take($this->filas);

        $map = [];
        
        if ($this->origenVenta === 'todos' || $this->origenVenta === 'factura') {
            $datosFacturasCfdi = DB::table(self::CFDI_TABLE)
                ->selectRaw(self::FECHA_CFDI_RAW . ' as periodo, SUM(c.total) as total');
            $this->aplicarFiltrosFacturas($datosFacturasCfdi);
            $datosFacturasCfdi = $datosFacturasCfdi->groupBy(DB::raw(self::FECHA_CFDI_RAW))->orderBy(DB::raw(self::FECHA_CFDI_RAW), 'asc')->get();
            
            foreach ($datosFacturasCfdi as $d) {
                $map[$d->periodo] = ($map[$d->periodo] ?? 0) + (float) $d->total;
            }
            
            $datosFacturasVentas = DB::table(self::VENTAS_TABLE)
                ->whereIn('v.idtipodocumento', [2, 10, 11, 12])
                ->selectRaw(self::FECHA_VENTA_RAW . ' as periodo, SUM(v.total) as total');
            $this->aplicarFiltrosVentas($datosFacturasVentas);
            $datosFacturasVentas = $datosFacturasVentas->groupBy(DB::raw(self::FECHA_VENTA_RAW))->orderBy(DB::raw(self::FECHA_VENTA_RAW), 'asc')->get();
            
            foreach ($datosFacturasVentas as $d) {
                $map[$d->periodo] = ($map[$d->periodo] ?? 0) + (float) $d->total;
            }
        }
        
        if ($this->origenVenta === 'todos' || $this->origenVenta === 'venta') {
            $datosVentasNotas = DB::table(self::VENTAS_TABLE)
                ->whereIn('v.idtipodocumento', [3, 7, 8, 9, 13, 14])
                ->selectRaw(self::FECHA_VENTA_RAW . ' as periodo, SUM(v.total) as total');
            $this->aplicarFiltrosVentas($datosVentasNotas);
            $datosVentasNotas = $datosVentasNotas->groupBy(DB::raw(self::FECHA_VENTA_RAW))->orderBy(DB::raw(self::FECHA_VENTA_RAW), 'asc')->get();
            
            foreach ($datosVentasNotas as $d) {
                $map[$d->periodo] = ($map[$d->periodo] ?? 0) + (float) $d->total;
            }
        }

        $periodos = collect($map)->map(function ($total, $periodo) {
            return (object) ['periodo' => $periodo, 'total' => $total];
        })->sortBy('periodo')->values();

        $datosGrafica = $periodos;

        $mapTipos = [];
        
        if ($this->origenVenta === 'todos' || $this->origenVenta === 'factura') {
            $tiposFact = DB::table(self::CFDI_TABLE)
                ->selectRaw("COALESCE(c.formadepago,'NO ESPECIFICADO') as tipopago, SUM(c.total) as total");
            $this->aplicarFiltrosFacturas($tiposFact);
            $tiposFact = $tiposFact->groupBy('c.formadepago')->get();
            
            foreach ($tiposFact as $t) {
                $mapTipos[$t->tipopago] = ($mapTipos[$t->tipopago] ?? 0) + (float) $t->total;
            }
            
            $tiposFactVentas = DB::table(self::VENTAS_TABLE)
                ->whereIn('v.idtipodocumento', [2, 10, 11, 12])
                ->selectRaw("COALESCE(v.tipopago,'NO ESPECIFICADO') as tipopago, SUM(v.total) as total");
            $this->aplicarFiltrosVentas($tiposFactVentas);
            $tiposFactVentas = $tiposFactVentas->groupBy('v.tipopago')->get();
            
            foreach ($tiposFactVentas as $t) {
                $mapTipos[$t->tipopago] = ($mapTipos[$t->tipopago] ?? 0) + (float) $t->total;
            }
        }
        
        if ($this->origenVenta === 'todos' || $this->origenVenta === 'venta') {
            $tiposVentNotas = DB::table(self::VENTAS_TABLE)
                ->whereIn('v.idtipodocumento', [3, 7, 8, 9, 13, 14])
                ->selectRaw("COALESCE(v.tipopago,'NO ESPECIFICADO') as tipopago, SUM(v.total) as total");
            $this->aplicarFiltrosVentas($tiposVentNotas);
            $tiposVentNotas = $tiposVentNotas->groupBy('v.tipopago')->get();
            
            foreach ($tiposVentNotas as $t) {
                $mapTipos[$t->tipopago] = ($mapTipos[$t->tipopago] ?? 0) + (float) $t->total;
            }
        }

        $ventasPorTipoPago = collect($mapTipos)->map(function ($total, $tipopago) {
            return (object) ['tipopago' => $tipopago, 'total' => $total];
        })->sortByDesc('total')->values();

        $totFact = DB::table(self::CFDI_TABLE);
        $this->aplicarFiltrosFacturas($totFact);
        
        $totVentFacturas = DB::table(self::VENTAS_TABLE)->whereIn('v.idtipodocumento', [2, 10, 11, 12]);
        $this->aplicarFiltrosVentas($totVentFacturas);
        
        $totVentNotas = DB::table(self::VENTAS_TABLE)->whereIn('v.idtipodocumento', [3, 7, 8, 9, 13, 14]);
        $this->aplicarFiltrosVentas($totVentNotas);

        if ($this->origenVenta === 'venta') {
            $totalVentas = $totVentNotas->count();
        } elseif ($this->origenVenta === 'factura') {
            $totalVentas = $totFact->count() + $totVentFacturas->count();
        } else {
            $totalVentas = $totFact->count() + $totVentFacturas->count() + $totVentNotas->count();
        }

        $subtotalFact = DB::table(self::CFDI_TABLE)->when(true, function ($q) { return $q; });
        $this->aplicarFiltrosFacturas($subtotalFact);
        $subtotalFactVal = (float) $subtotalFact->sum('c.subtotaldespuesdesc');

        $subtotalVent = DB::table(self::VENTAS_TABLE);
        $this->aplicarFiltrosVentas($subtotalVent);
        $subtotalVentVal = (float) $subtotalVent->sum('v.subtotal');

        $subtotalVentas = $subtotalFactVal + $subtotalVentVal;
        $totalComisiones = 0;
        $totalGeneral = (float) ($subtotalFactVal + $subtotalVentVal);

        $sucursales = Sucursal::where('activo', 1)->get();

        $usuariosFact = DB::table('cfdi_general01')->select('usuario')->distinct()->whereNotNull('usuario')->where('usuario', '!=', '')->get()->pluck('usuario');
        $usuariosVent = DB::table('ventasgeneral')->select('usuario')->distinct()->whereNotNull('usuario')->where('usuario', '!=', '')->get()->pluck('usuario');
        $usuarios = $usuariosFact->merge($usuariosVent)->unique()->sort()->values()->map(function ($u) { return (object) ['usuario' => $u]; });

        $tiposFactPago = DB::table('cfdi_general01')->select('formadepago')->distinct()->whereNotNull('formadepago')->get()->pluck('formadepago');
        $tiposVentPago = DB::table('ventasgeneral')->select('tipopago')->distinct()->whereNotNull('tipopago')->get()->pluck('tipopago');
        $tiposPago = $tiposFactPago->merge($tiposVentPago)->unique()->sort()->values()->map(function ($t) { return (object) ['formadepago' => $t]; });

        return view('livewire.reportes.ventaspor.index', [
            'ventas' => $ventas,
            'datosGrafica' => $datosGrafica,
            'ventasPorTipoPago' => $ventasPorTipoPago,
            'totalVentas' => $totalVentas,
            'subtotalVentas' => $subtotalVentas,
            'totalComisiones' => $totalComisiones,
            'totalGeneral' => $totalGeneral,
            'sucursales' => $sucursales,
            'usuarios' => $usuarios,
            'tiposPago' => $tiposPago,
            'origenOpciones' => ['todos' => 'Todos', 'factura' => 'Ventas por factura', 'venta' => 'Otro tipo de ventas'],
        ]);
    }
}

