<?php

namespace App\Livewire\Ventas\Cotizaciones;

use Illuminate\Support\Facades\Log;

class CotizacionesPartidaHandler
{
    private const DEFAULT_IVA = '16';
    private const DEFAULT_RETENCION_IVA = '10.67';

    private static function bcround(string $number, int $precision = 2): string
    {
        $scale = max(0, $precision) + 1;
        $increment = '0.' . str_repeat('0', max(0, $precision)) . '5';
        $comp = bccomp($number, '0', $scale);
        $adjusted = $comp >= 0
            ? bcadd($number, $increment, $scale)
            : bcsub($number, $increment, $scale);
        return bcadd($adjusted, '0', max(0, $precision));
    }

    private static function bcclamp(string $value, string $min, string $max, int $scale = 8): string
    {
        if (bccomp($value, $min, $scale) < 0) { return $min; }
        if (bccomp($value, $max, $scale) > 0) { return $max; }
        return $value;
    }
    public static function inicializarPartidaVacia(): array
    {
        return [
            'cantidad' => 1,
            'precio_unitario' => 0.0,
            'importe' => 0.0,
            'descuento1_porcentaje' => 0.0,
            'descuento1_importe' => 0.0,
            'descuento2_porcentaje' => 0.0,
            'descuento2_importe' => 0.0,
            'descuento3_porcentaje' => 0.0,
            'descuento3_importe' => 0.0,
            'descuento_total' => 0.0,
            'subtotal' => 0.0,
            'iva_porcentaje' => (float) self::DEFAULT_IVA,
            'iva_importe' => 0.0,
            'ieps_porcentaje' => 0.0,
            'ieps_importe' => 0.0,
            'retencion_isr_porcentaje' => 0.0,
            'retencion_isr_importe' => 0.0,
            'retencion_iva_porcentaje' => (float) self::DEFAULT_RETENCION_IVA,
            'retencion_iva_importe' => 0.0,
            'modalidad_ieps' => 'exento',
            'importe_total' => 0.0,
            'descripcion_adicional' => '',
            'clave' => '',
            'descripcion' => '',
        ];
    }


    public static function prepararPartidaParaEdicion(array $partida): array
    {
        $get = static fn($arr, $k, $alt = null) => $arr[$k] ?? $alt;
        return [
            'clave' => $get($partida, 'clave', ''),
            'descripcion' => $get($partida, 'descripcion', ''),
            'cantidad' => (int) ($get($partida, 'cantidad', 1) ?? 1),
            'precio_unitario' => (float) self::bcround((string) ($get($partida, 'precio_unitario', '0') ?? '0'), 2),
            'importe' => (float) self::bcround((string) ($get($partida, 'importe', '0') ?? '0'), 2),

            'descuento1_porcentaje' => (float) self::bcround((string) ($get($partida, 'descuento1_porcentaje', '0') ?? '0'), 2),
            'descuento1_importe' => (float) self::bcround((string) ($get($partida, 'descuento1_importe', '0') ?? '0'), 2),
            'descuento2_porcentaje' => (float) self::bcround((string) ($get($partida, 'descuento2_porcentaje', '0') ?? '0'), 2),
            'descuento2_importe' => (float) self::bcround((string) ($get($partida, 'descuento2_importe', '0') ?? '0'), 2),
            'descuento3_porcentaje' => (float) self::bcround((string) ($get($partida, 'descuento3_porcentaje', '0') ?? '0'), 2),
            'descuento3_importe' => (float) self::bcround((string) ($get($partida, 'descuento3_importe', '0') ?? '0'), 2),
            'descuento_total' => (float) self::bcround((string) ($get($partida, 'descuento_total', $get($partida, 'descuento', '0')) ?? '0'), 2),

            'subtotal' => (float) self::bcround((string) ($get($partida, 'subtotal', '0') ?? '0'), 2),

            'iva_porcentaje' => (float) self::bcround((string) ($get($partida, 'iva_porcentaje', self::DEFAULT_IVA) ?? self::DEFAULT_IVA), 2),
            'iva_importe' => (float) self::bcround((string) ($get($partida, 'iva_importe', $get($partida, 'iva_trasladado', '0')) ?? '0'), 2),
            'ieps_porcentaje' => (float) self::bcround((string) ($get($partida, 'ieps_porcentaje', '0') ?? '0'), 2),
            'ieps_importe' => (float) self::bcround((string) ($get($partida, 'ieps_importe', $get($partida, 'ieps_trasladado', '0')) ?? '0'), 2),
            'retencion_isr_porcentaje' => (float) self::bcround((string) ($get($partida, 'retencion_isr_porcentaje', $get($partida, 'isr_retenido_porcentaje', '0')) ?? '0'), 2),
            'retencion_isr_importe' => (float) self::bcround((string) ($get($partida, 'retencion_isr_importe', $get($partida, 'isr_retenido', '0')) ?? '0'), 2),
            'retencion_iva_porcentaje' => (float) self::bcround((string) ($get($partida, 'retencion_iva_porcentaje', self::DEFAULT_RETENCION_IVA) ?? self::DEFAULT_RETENCION_IVA), 2),
            'retencion_iva_importe' => (float) self::bcround((string) ($get($partida, 'retencion_iva_importe', $get($partida, 'iva_retenido', '0')) ?? '0'), 2),
            'modalidad_ieps' => $get($partida, 'modalidad_ieps', 'exento'),

            'importe_total' => (float) self::bcround((string) ($get($partida, 'importe_total', $get($partida, 'total', '0')) ?? '0'), 2),
            'descripcion_adicional' => $get($partida, 'descripcion_adicional', ''),
        ];
    }

    public static function recalcularPartida(array $partidaEditando): array
    {
        if (!$partidaEditando) {
            return self::inicializarPartidaVacia();
        }

    $precision = 8;
    $cantidad = (string) max(1, (int) ($partidaEditando['cantidad'] ?? 1));
    $precioUnitario = (string) max(0, (float) ($partidaEditando['precio_unitario'] ?? 0));

    $desc1_porcentaje = self::bcclamp((string) ($partidaEditando['descuento1_porcentaje'] ?? '0'), '0', '100');
    $desc2_porcentaje = self::bcclamp((string) ($partidaEditando['descuento2_porcentaje'] ?? '0'), '0', '100');
    $desc3_porcentaje = self::bcclamp((string) ($partidaEditando['descuento3_porcentaje'] ?? '0'), '0', '100');
    $iva_porcentaje = self::bcclamp((string) ($partidaEditando['iva_porcentaje'] ?? self::DEFAULT_IVA), '0', '100');
    $ieps_porcentaje = (string) max(0, (float) ($partidaEditando['ieps_porcentaje'] ?? 0));
    $retencion_isr_porcentaje = self::bcclamp((string) ($partidaEditando['retencion_isr_porcentaje'] ?? '0'), '0', '100');
    $retencion_iva_porcentaje = self::bcclamp((string) ($partidaEditando['retencion_iva_porcentaje'] ?? self::DEFAULT_RETENCION_IVA), '0', '100');
        
    $partidaEditando['cantidad'] = (int) $cantidad;
    $partidaEditando['precio_unitario'] = (float) self::bcround($precioUnitario, 2);
    $partidaEditando['descuento1_porcentaje'] = (float) self::bcround($desc1_porcentaje, 2);
    $partidaEditando['descuento2_porcentaje'] = (float) self::bcround($desc2_porcentaje, 2);
    $partidaEditando['descuento3_porcentaje'] = (float) self::bcround($desc3_porcentaje, 2);
    $partidaEditando['iva_porcentaje'] = (float) self::bcround($iva_porcentaje, 2);
    $partidaEditando['ieps_porcentaje'] = (float) self::bcround($ieps_porcentaje, 2);
    $partidaEditando['retencion_isr_porcentaje'] = (float) self::bcround($retencion_isr_porcentaje, 2);
    $partidaEditando['retencion_iva_porcentaje'] = (float) self::bcround($retencion_iva_porcentaje, 2);
        
    $importe = bcmul($cantidad, $precioUnitario, $precision);
    $partidaEditando['importe'] = (float) self::bcround($importe, 2);

    $baseDescuento1 = $importe;
    $d1 = bcmul($baseDescuento1, bcdiv($desc1_porcentaje, '100', $precision), $precision);
    $partidaEditando['descuento1_importe'] = (float) self::bcround($d1, 2);
        
    $baseDescuento2 = bcsub($baseDescuento1, $d1, $precision);
    $d2 = bcmul($baseDescuento2, bcdiv($desc2_porcentaje, '100', $precision), $precision);
    $partidaEditando['descuento2_importe'] = (float) self::bcround($d2, 2);
        
    $baseDescuento3 = bcsub($baseDescuento2, $d2, $precision);
    $d3 = bcmul($baseDescuento3, bcdiv($desc3_porcentaje, '100', $precision), $precision);
    $partidaEditando['descuento3_importe'] = (float) self::bcround($d3, 2);
        
    $descTotal = bcadd(bcadd($d1, $d2, $precision), $d3, $precision);
    $partidaEditando['descuento_total'] = (float) self::bcround($descTotal, 2);
        
    $subtotal = bcsub($importe, $descTotal, $precision);
    if (bccomp($subtotal, '0', $precision) < 0) { $subtotal = '0'; }
    $partidaEditando['subtotal'] = (float) self::bcround($subtotal, 2);
        
    $iva_imp = bcmul($subtotal, bcdiv($iva_porcentaje, '100', $precision), $precision);
    $ieps_imp = bcmul($subtotal, bcdiv($ieps_porcentaje, '100', $precision), $precision);
    $ret_isr_imp = bcmul($subtotal, bcdiv($retencion_isr_porcentaje, '100', $precision), $precision);
    $ret_iva_imp = bcmul($subtotal, bcdiv($retencion_iva_porcentaje, '100', $precision), $precision);
    $partidaEditando['iva_importe'] = (float) self::bcround($iva_imp, 2);
    $partidaEditando['ieps_importe'] = (float) self::bcround($ieps_imp, 2);
    $partidaEditando['retencion_isr_importe'] = (float) self::bcround($ret_isr_imp, 2);
    $partidaEditando['retencion_iva_importe'] = (float) self::bcround($ret_iva_imp, 2);
        
        $total = bcsub(
            bcadd(bcadd($subtotal, $iva_imp, $precision), $ieps_imp, $precision),
            bcadd($ret_isr_imp, $ret_iva_imp, $precision),
            $precision
        );
        $partidaEditando['importe_total'] = (float) self::bcround($total, 2);

        return $partidaEditando;
    }

    public static function actualizarPartidaEnProductos(array &$productosagregados, int $indice, array $partidaEditando): bool
    {
        if (!isset($productosagregados[$indice])) {
            return false;
        }

        $partida = &$productosagregados[$indice];
        
        $calculosUnificados = CotizacionesBaseLogic::sincronizarCalculosModal($partidaEditando);
        
        $partida['cantidad'] = $partidaEditando['cantidad'];
        $partida['precio_unitario'] = $partidaEditando['precio_unitario'];
        $partida['importe'] = $calculosUnificados['importe'];
        $partida['descuento'] = $partidaEditando['descuento_total'];
        $partida['subtotal'] = $calculosUnificados['subtotal'];
        $partida['iva_trasladado'] = $calculosUnificados['iva_trasladado'];
        $partida['ieps_trasladado'] = $calculosUnificados['ieps_trasladado'];
        $partida['iva_retenido'] = $calculosUnificados['iva_retenido'];
        $partida['isr_retenido'] = $calculosUnificados['isr_retenido'];
        $partida['total'] = $calculosUnificados['total'];
        $partida['descripcion_adicional'] = $partidaEditando['descripcion_adicional'];
        $partida['iva_porcentaje'] = $calculosUnificados['iva_porcentaje'];
        $partida['iva_retenido_porcentaje'] = $calculosUnificados['iva_retenido_porcentaje'];
        $partida['ieps_porcentaje'] = $calculosUnificados['ieps_porcentaje'];
        $partida['isr_retenido_porcentaje'] = $calculosUnificados['isr_retenido_porcentaje'];
        $partida['descuento1_porcentaje'] = $partidaEditando['descuento1_porcentaje'];
        $partida['descuento1_importe'] = $partidaEditando['descuento1_importe'];
        $partida['descuento2_porcentaje'] = $partidaEditando['descuento2_porcentaje'];
        $partida['descuento2_importe'] = $partidaEditando['descuento2_importe'];
        $partida['descuento3_porcentaje'] = $partidaEditando['descuento3_porcentaje'];
        $partida['descuento3_importe'] = $partidaEditando['descuento3_importe'];
        $partida['descuento_total'] = $partidaEditando['descuento_total'];
        $partida['iva_importe'] = $partidaEditando['iva_importe'];
        $partida['ieps_importe'] = $partidaEditando['ieps_importe'];
        $partida['retencion_isr_porcentaje'] = $partidaEditando['retencion_isr_porcentaje'];
        $partida['retencion_isr_importe'] = $partidaEditando['retencion_isr_importe'];
        $partida['retencion_iva_porcentaje'] = $partidaEditando['retencion_iva_porcentaje'];
        $partida['retencion_iva_importe'] = $partidaEditando['retencion_iva_importe'];
        $partida['modalidad_ieps'] = $partidaEditando['modalidad_ieps'];
        $partida['importe_total'] = $partidaEditando['importe_total'];

        return true;
    }

    public static function validarPartida(array $partida): array
    {
        $errores = [];

        $cantidad = floatval($partida['cantidad'] ?? 0);
        if ($cantidad <= 0) {
            $errores[] = 'La cantidad debe ser mayor a 0';
        }

        $precioUnitario = floatval($partida['precio_unitario'] ?? 0);
        if ($precioUnitario < 0) {
            $errores[] = 'El precio unitario debe ser mayor o igual a 0';
        }

        if (empty($partida['clave'])) {
            $errores[] = 'La clave del producto es requerida';
        }

        if (empty($partida['descripcion'])) {
            $errores[] = 'La descripción del producto es requerida';
        }

        for ($i = 1; $i <= 3; $i++) {
            $porcentaje = floatval($partida["descuento{$i}_porcentaje"] ?? 0);
            if ($porcentaje < 0 || $porcentaje > 100) {
                $errores[] = "El descuento {$i} debe estar entre 0 y 100%";
            }
        }

        $iva_porcentaje = floatval($partida['iva_porcentaje'] ?? 0);
        if ($iva_porcentaje < 0 || $iva_porcentaje > 100) {
            $errores[] = 'El porcentaje de IVA debe estar entre 0 y 100%';
        }

        $ieps_porcentaje = floatval($partida['ieps_porcentaje'] ?? 0);
        if ($ieps_porcentaje < 0) {
            $errores[] = 'El porcentaje de IEPS debe ser mayor o igual a 0';
        }

        $retencion_iva_porcentaje = floatval($partida['retencion_iva_porcentaje'] ?? 0);
        if ($retencion_iva_porcentaje < 0 || $retencion_iva_porcentaje > 100) {
            $errores[] = 'El porcentaje de retención de IVA debe estar entre 0 y 100%';
        }

        $retencion_isr_porcentaje = floatval($partida['retencion_isr_porcentaje'] ?? 0);
        if ($retencion_isr_porcentaje < 0 || $retencion_isr_porcentaje > 100) {
            $errores[] = 'El porcentaje de retención de ISR debe estar entre 0 y 100%';
        }

        return $errores;
    }

    public static function iniciarEdicionPartida(array $productosagregados, int $indice): array
    {
        if (!isset($productosagregados[$indice])) {
            return [
                'success' => false,
                'message' => 'Partida no encontrada'
            ];
        }

        return [
            'success' => true,
            'partida' => self::prepararPartidaParaEdicion($productosagregados[$indice]),
            'indice' => $indice
        ];
    }

    public static function confirmarEdicionPartida(array &$productosagregados, int $indice, array $partidaEditando): array
    {
        Log::info('Confirmando edición de partida:', [
            'indice' => $indice,
            'partidaEditando' => $partidaEditando
        ]);

        $errores = self::validarPartida($partidaEditando);
        
        if (!empty($errores)) {
            return [
                'success' => false,
                'errores' => $errores
            ];
        }

        $actualizado = self::actualizarPartidaEnProductos($productosagregados, $indice, $partidaEditando);

        if ($actualizado) {
            Log::info('Partida actualizada:', [
                'producto_actualizado' => $productosagregados[$indice]
            ]);
            
            return [
                'success' => true,
                'message' => 'Partida actualizada correctamente.'
            ];
        } else {
            return [
                'success' => false,
                'message' => 'Error al actualizar la partida.'
            ];
        }
    }

    public static function resetearModalPartida(): array
    {
        return [
            'mostrarModalPartida' => false,
            'indicePartidaEditando' => null,
            'partidaEditando' => []
        ];
    }

    public static function debugPartida(array $partida): void
    {
        Log::info('Debug Partida:', [
            'cantidad' => $partida['cantidad'] ?? 'N/A',
            'precio_unitario' => $partida['precio_unitario'] ?? 'N/A',
            'importe' => $partida['importe'] ?? 'N/A',
            'descuento' => $partida['descuento'] ?? 'N/A',
            'descuento_total' => $partida['descuento_total'] ?? 'N/A',
            'subtotal' => $partida['subtotal'] ?? 'N/A',
            'total' => $partida['total'] ?? 'N/A',
            'descuento1_porcentaje' => $partida['descuento1_porcentaje'] ?? 'N/A',
            'descuento2_porcentaje' => $partida['descuento2_porcentaje'] ?? 'N/A',
            'descuento3_porcentaje' => $partida['descuento3_porcentaje'] ?? 'N/A',
        ]);
    }
}
