<?php

namespace App\Livewire\Ventas\Cotizaciones;

use App\Models\Sucursal;
use Illuminate\Support\Facades\Auth;
use Livewire\Attributes\Title;
use Livewire\Component;
use Mpdf\Mpdf;

#[Title('Cotizaciones')]
class Index extends Component
{

    public $search = [
        'clientes' => '',
        'productos' => '',
    ];
    
    public $nombre = '';
    public $telefono = '';
    public $clave = '';
    public $rfc = '';
    public $fechaForm = '';
    public $colonia = '';
    public $sucursalid;
    public $serie = 'NB';
    public $fecha;
    public $sucursales = [];
    public $idclient = '';
    public $clientesearch = '';
    public $clientenombre = '';
    public $clienterfc = '';
    public $clientedireccion = '';
    public $productoid;
    public $productosearch = '';
    public $cantidad = 1;
    public $precioventaantesiva = 0;
    public $precioventaconiva = 0;
    public $descuento = 0;
    public $descripcion = '';
    public $descripcionadicional = '';
    public $productos = [];
    public $claveproductoservicio = '';
    public $descripcionclaveproductoservicio = '';
    public $claveunidadmedida = '';
    public $descripcionunidadmedida = '';
    public $ivatrasladado = 0;
    public $ivaretenido = 0;
    public $iepstrasladado = 0;
    public $isrretenido = 0;
    public $productosagregados = [];
    public $importetotal = 0;
    public $descuentototal = 0;
    public $subtotaltotal = 0;
    public $retivatotal = 0;
    public $retisrtotal = 0;
    public $iepstotal = 0;
    public $impuestototal = 0;
    public $totalfinal = 0;
    public $importetotalletras = 'CERO PESOS';
    public $idempresa;
    public $selectedCliente = null;
    public $selectedProducto = null;
    public $mostrarTablaClientes = false;
    public $mostrarTablaProductos = false;
    public $folio;
    
    public $mostrarModalPartida = false;
    public $indicePartidaEditando = null;
    public $partidaEditando = [];

    public function mount()
    {
        $this->idempresa = Auth::user()->idempresa;
        $this->fecha = now()->format('Y-m-d');
        $this->fechaForm = now()->format('d-m-Y');
        $this->cargarSucursales();
        $this->sucursalid = $this->sucursales[0]['id'] ?? null;
        $this->generarFolio();
    }

    public function cargarSucursales()
    {
        $this->sucursales = Sucursal::where('idempresa', Auth::user()->idempresa)
            ->orderBy('razon_social')
            ->pluck('razon_social', 'idsucursal')
            ->toArray();
    }

    public function generarFolio()
    {
        $fecha = now()->format('Ymd');
        $consecutivo = str_pad(rand(1, 9999), 4, '0', STR_PAD_LEFT);
        $this->folio = $this->serie . '-' . $fecha . '-' . $consecutivo;
    }

    private function obtenerClientes()
    {
        return CotizacionesDataHandler::obtenerClientes(
            $this->idempresa,
            $this->search['clientes'],
            true
        );
    }

    private function obtenerProductos()
    {
        return CotizacionesDataHandler::obtenerProductos(
            $this->idempresa,
            $this->search['productos'],
            true
        );
    }

    public function buscarClientes()
    {
        $this->mostrarTablaClientes = true;
        $this->selectedCliente = null;
    }

    public function buscarProductos()
    {
        $this->mostrarTablaProductos = true;
        $this->selectedProducto = null;
        
        $productos = CotizacionesDataHandler::obtenerProductos($this->idempresa, $this->search['productos'], true);
        
        $this->productos = is_array($productos) ? collect($productos) : $productos;
        
        if (count($this->productos) === 0) {
            session()->flash('message', 'No se encontraron productos para la búsqueda realizada.');
        }
    }

    public function updatedSearch($value, $key)
    {
        if ($key === 'clientes') {
            $this->mostrarTablaClientes = true;
            $this->mostrarTablaProductos = false;
        }
        
        if ($key === 'productos') {
            $this->mostrarTablaProductos = true;
            $this->mostrarTablaClientes = false;
            $this->productos = $this->obtenerProductos();
        }
    }

    public function abrirDropdownClientes()
    {
        $this->mostrarTablaClientes = true;
        $this->mostrarTablaProductos = false;
    }

    public function abrirDropdownProductos()
    {
        $this->mostrarTablaProductos = true;
        $this->productos = $this->obtenerProductos();
    }

    public function cerrarDropdownClientes()
    {
        $this->mostrarTablaClientes = false;
    }

    public function cerrarDropdownProductos()
    {
        $this->mostrarTablaProductos = false;
    }

    public function confirmarSeleccionCliente()
    {
        if ($this->selectedCliente) {
            $datos = CotizacionesDataHandler::procesarSeleccionCliente($this->selectedCliente);
            
            foreach ($datos as $propiedad => $valor) {
                $this->$propiedad = $valor;
            }
            
            $this->mostrarTablaClientes = false;
            $this->selectedCliente = null;
        }
    }

    public function confirmarSeleccionProducto()
    {
        if ($this->selectedProducto) {
            $datos = CotizacionesDataHandler::procesarSeleccionProducto($this->selectedProducto);
            
            foreach ($datos as $propiedad => $valor) {
                $this->$propiedad = $valor;
            }
            
            $this->calcularImpuestos();
            $this->mostrarTablaProductos = false;
            $this->selectedProducto = null;
        }
    }

    public function seleccionarCliente($clienteId)
    {
        $cliente = \App\Models\Cliente::find($clienteId);
        if ($cliente) {
            $this->idclient = $cliente->idclient;
            $this->clientenombre = $cliente->nombre;
            $this->clienterfc = $cliente->rfc ?? '';
            $this->clientedireccion = $cliente->direccion_completa ?? '';
            $this->search['clientes'] = $cliente->nombre;
            $this->mostrarTablaClientes = false;
        }
    }

    public function seleccionarProducto($productoId)
    {
        $producto = \App\Models\Producto::find($productoId);
        if ($producto) {
            $this->productoid = $producto->idproduct;
            $this->productosearch = $producto->clave . ' - ' . $producto->descripcion;
            $this->descripcion = $producto->descripcion;
            $this->precioventaantesiva = $producto->precioventaantesiva;
            $this->search['productos'] = $producto->clave . ' - ' . $producto->descripcion;
            $this->mostrarTablaProductos = false;
            
            $this->claveproductoservicio = $producto->clave_prodserv ?? '';
            $this->descripcionclaveproductoservicio = $producto->prodserv ?? '';
            $this->claveunidadmedida = $producto->clave_unidadmedida ?? '';
            $this->descripcionunidadmedida = $producto->unidadmedida ?? '';
            
            $this->calcularImpuestos();
        }
    }

    public function calcularImpuestos()
    {
        $impuestos = CotizacionesBaseLogic::calcularImpuestos(
            $this->cantidad,
            $this->precioventaantesiva,
            $this->descuento
        );
        
        $this->ivatrasladado = $impuestos['ivatrasladado'];
        $this->ivaretenido = $impuestos['ivaretenido'];
        $this->iepstrasladado = $impuestos['iepstrasladado'];
        $this->isrretenido = $impuestos['isrretenido'];
        $this->precioventaconiva = $impuestos['precioConIva'];
    }

    public function updated($name)
    {
        if (in_array($name, ['cantidad', 'precioventaantesiva', 'descuento'])) {
            $this->calcularImpuestos();
        }
        
        if (str_starts_with($name, 'partidaEditando.')) {
            $this->partidaEditando = CotizacionesPartidaHandler::recalcularPartida($this->partidaEditando);
        }
    }

    public function agregarProducto()
    {
        $this->validate([
            'productoid' => 'required',
            'cantidad' => 'required|numeric|min:1',
            'precioventaantesiva' => 'required|numeric|min:0',
        ]);

        $producto = CotizacionesBaseLogic::crearProductoParaCotizacion([
            'productoid' => $this->productoid,
            'clave' => explode(' - ', $this->productosearch)[0],
            'descripcion' => $this->descripcion,
            'cantidad' => $this->cantidad,
            'precio' => $this->precioventaantesiva,
            'descuento' => $this->descuento,
            'descripcion_adicional' => $this->descripcionadicional,
            'clave_unidadmedida' => $this->claveunidadmedida
        ]);

        $this->productosagregados[] = $producto;
        $this->calcularTotales();
        $this->limpiarCamposProducto();
    }

    public function eliminarProducto($id)
    {
        $this->productosagregados = array_filter($this->productosagregados, function ($producto) use ($id) {
            return $producto['id'] !== $id;
        });
        $this->calcularTotales();
    }

    public function calcularTotales()
    {
        $totales = CotizacionesBaseLogic::calcularTotalesCotizacion($this->productosagregados);
        
        $this->importetotal = $totales['importetotal'];
        $this->descuentototal = $totales['descuentototal'];
        $this->subtotaltotal = $totales['subtotaltotal'];
        $this->retivatotal = $totales['retivatotal'];
        $this->retisrtotal = $totales['retisrtotal'];
        $this->iepstotal = $totales['iepstotal'];
        $this->impuestototal = $totales['impuestototal'];
        $this->totalfinal = $totales['totalfinal'];

        $this->importetotalletras = CotizacionesBaseLogic::numeroALetras($this->totalfinal);
    }

    public function limpiarCamposProducto()
    {
        $campos = CotizacionesBaseLogic::limpiarCamposProducto();
        foreach ($campos as $campo => $valor) {
            $this->$campo = $valor;
        }
        
        $this->selectedProducto = null;
        $this->search['productos'] = '';
        $this->mostrarTablaProductos = false;
    }

    public function cancelarProducto()
    {
        $this->limpiarCamposProducto();
        
        $this->mostrarTablaProductos = false;
        
        $this->search['productos'] = '';
    }

    public function resetModal()
    {
        $this->selectedCliente = null;
        $this->search['clientes'] = '';
        $this->mostrarTablaClientes = false;
        $this->mostrarTablaProductos = false;
    }

    public function guardarCotizacion()
    {
        $this->validate([
            'idclient' => 'required',
            'productosagregados' => 'required|array|min:1',
        ]);

        if (!CotizacionesBaseLogic::validarCalculos($this->productosagregados)) {
            $this->addError('general', 'Error en los cálculos. Por favor, verifique los datos.');
            return null;
        }

        $cliente = \App\Models\Cliente::find($this->idclient);
        if (!$cliente) {
            $this->addError('general', 'Cliente no encontrado.');
            return null;
        }

        $sucursal = Sucursal::find($this->sucursalid);

        $datos = [
            'folio' => $this->folio,
            'fecha' => now()->format('d/m/Y'),
            'empresa' => [
                'razon_social' => $sucursal->razon_social ?? 'RAZON SOCIAL DE LA EMPRESA DEM',
                'rfc' => 'SAMJ770624411',
                'direccion' => 'CALLE 59 2448 X 118 Y 120 FRACC. YUCALPETEN',
                'ciudad' => 'MERIDA YUCATAN MEXICO',
                'cp' => '97248',
                'lugar_expedicion' => '97248 YUCATAN',
                'regimen_fiscal' => '621|REGIMEN DE INCORPORACION FISCAL',
                'telefono' => '01.999.945.99.99'
            ],
            'cliente' => [
                'nombre' => $cliente->nombre,
                'rfc' => $cliente->rfc,
                'direccion' => $cliente->direccion_completa,
                'cp' => $cliente->codigopostal,
                'regimen_fiscal' => '621|Incorporación Fiscal'
            ],
            'productos' => collect($this->productosagregados)->map(function($producto) {
                return [
                    'clave' => $producto['clave'] ?? $producto['producto_id'] ?? '',
                    'cantidad' => $producto['cantidad'],
                    'unidad' => $producto['clave_unidadmedida'],
                    'descripcion' => $producto['descripcion'],
                    'precio_unitario' => $producto['precio_unitario'],
                    'importe' => $producto['importe'],
                    'subtotal' => $producto['subtotal'],
                    'total' => $producto['total']
                ];
            })->toArray(),
            'totales' => [
                'subtotal' => $this->subtotaltotal,
                'iva' => $this->impuestototal,
                'ieps' => $this->iepstotal,
                'retencion_isr' => $this->retisrtotal,
                'retencion_iva' => $this->retivatotal,
                'retenciones' => $this->retisrtotal + $this->retivatotal,
                'total' => $this->totalfinal
            ],
            'total_letras' => $this->importetotalletras
        ];

        $html = view('exports.cotizacion', $datos)->render();

        $mpdf = new Mpdf([
            'mode' => 'utf-8',
            'format' => 'A4',
            'orientation' => 'P'
        ]);

        $mpdf->WriteHTML($html);

        session()->flash('message', 'Cotización guardada exitosamente.');

        return response()->streamDownload(function() use ($mpdf) {
            $mpdf->Output('cotizacion.pdf', 'D');
        }, 'cotizacion_' . date('Ymd_His') . '.pdf');
    }

    public function editarPartida($indice)
    {
        $resultado = CotizacionesPartidaHandler::iniciarEdicionPartida($this->productosagregados, $indice);
        
        if ($resultado['success']) {
            $this->indicePartidaEditando = $resultado['indice'];
            $this->partidaEditando = $resultado['partida'];
            $this->mostrarModalPartida = true;
        }
    }

    public function cancelarEdicionPartida()
    {
        $estado = CotizacionesPartidaHandler::resetearModalPartida();
        foreach ($estado as $propiedad => $valor) {
            $this->$propiedad = $valor;
        }
    }

    public function confirmarEdicionPartida()
    {
        if ($this->indicePartidaEditando === null) {
            return;
        }

        $resultado = CotizacionesPartidaHandler::confirmarEdicionPartida(
            $this->productosagregados,
            $this->indicePartidaEditando,
            $this->partidaEditando
        );

        if ($resultado['success']) {
            $this->calcularTotales();
            $this->cancelarEdicionPartida();
            session()->flash('message', $resultado['message']);
        } else {
            if (isset($resultado['errores'])) {
                foreach ($resultado['errores'] as $error) {
                    $this->addError('partidaEditando', $error);
                }
            } else {
                $this->addError('general', $resultado['message']);
            }
        }
    }

    public function cancelar()
    {
        $this->reset();
        $this->mount();
        session()->flash('message', 'Operación cancelada.');
    }

    public function render()
    {
        return view('livewire.ventas.cotizaciones.index', [
            'clientes' => $this->obtenerClientes(),
            'productos' => $this->productos,
        ]);
    }
}
