<?php

namespace App\Repositories\Inventarios;

use App\Enums\Status;
use App\Http\Requests\Inventarios\RegistrarMovimientoInventarioRequest;
use App\Models\MovimientoInventarioGeneral;
use App\Models\MovimientoInventarioDetalle;
use App\Models\Producto;
use App\Models\Usuario;
use App\Support\StringUtils;
use Carbon\Carbon;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use Illuminate\Validation\ValidationException;

class InventariosRepository implements InventariosRepositoryInterface
{
    use \App\Traits\WithFormRequest;

    public function registrar(array $data): bool
    {
        $this->validateWith(RegistrarMovimientoInventarioRequest::class, $data);
        $partidas = $data['partidas'];
        $almacen = \App\Models\Almacen::with('sucursal')->findOrFail($data['almacen_id']);
        $conceptoInv = \App\Models\Concepto::findOrFail($data['concepto_id']);
        $proveedor = null;
        if ($data['proveedor_id']) {
            $proveedor = \App\Models\Proveedor::findOrFail($data['proveedor_id']);
        }
        $costoInventario = $data['importe_total'];
        $tipoMovimiento = strtoupper($data['tipo_movimiento']);
        $fecha = $data['fecha_movimiento'];

        $actualDate = Carbon::now();
        $nombreProveedor = $proveedor->nombre ?? '';
        
        // Obtener el primer usuario activo de la sucursal
        $user = Usuario::where('idsucursal', $almacen->idsucursal)
            ->first();
        
        if (!$user) {
            throw ValidationException::withMessages(['almacen_id' => 'No se encontró un usuario válido para registrar movimientos en esta sucursal.']);
        }
        $movInvGeneral = MovimientoInventarioGeneral::create([
            'idsucursal'             => $almacen->idsucursal,
            'idinventariofisicogral' => 0,
            'fecha'                  => $fecha,
            'idalmacen'              => $almacen->idwarehouse,
            'estatus'                => Status::CERRADA,
            'idconceptoinventario'   => $conceptoInv->idinventoryconcept,
            'idusuario'              => $user->iduser,
            'costoinventario'        => $costoInventario,
            'idalmacendestino'       => 0, /*No hay destino; simple movimiento de ESTRADA*/
            'idorigeninventario'     => 3, /*valor => cat_origeninventario.idinventorysource = INVENTARIO FISICO*/
            'comentarios'            => '(TIPO MOVIMIENTO=INVENTARIO) - MOVIMIENTO CORRESPONDE A ' .
                $conceptoInv->conceptoinventario,
            'horamovimiento'         => $actualDate->format('H:i:s'),
            'nombre_proveedor'       => $nombreProveedor,
            'idturno'                => 0, /*No hay turno en web*/
            'fechahora_registro'     => $actualDate,
            'usuario'                => $user->nombre . ' - WEB',
            'from_host'              => 'SERVIDOR WEB',
            'origen_movimiento'      => $tipoMovimiento . ' POR INVENTARIOS',
            'tipo_movimiento'        => $tipoMovimiento,
            'signo'                  => $conceptoInv->signo,
            'origen_registro'        => 'WEB',
            'descargado'             => 0,
            'idproveedor'            => $proveedor->idprovider ?? null,
            'idmovinventariogral_desk' => 0,
        ]);
        if (!$movInvGeneral) {
            throw ValidationException::withMessages(['error' => 'Error al registrar el movimiento de inventario.']);
        }

        $this->registrarMovimientosDetalle($partidas, $almacen, $movInvGeneral, $conceptoInv, $actualDate);

        return true;
    }

    private function registrarMovimientosDetalle(array $partidas, $almacen, $movInvGeneral, $conceptoInv, $actualDate): void
    {
        foreach ($partidas as $partida) {
            $justificacion = $partida['justificacion'] ?? '';
            if (StringUtils::isEmpty($justificacion)) {
                $justificacion = $conceptoInv->conceptoinventario;
            }

            $producto = Producto::with(['familia', 'marca'])
                ->find($partida['producto_id']);
            if (!$producto) {
                throw ValidationException::withMessages(['error' => 'No se encontró el producto con ID ' . ($partida['producto_id'] ?? 'sin datos')]);
            }

            $movDetalle = new MovimientoInventarioDetalle();
            $movDetalle->idsucursal = $almacen->idsucursal;
            $movDetalle->idinventariofisicodet = 0;
            $movDetalle->idinventariofisicogral = $movInvGeneral->idmovinventariogral;
            $movDetalle->idproducto = $producto->idproduct;
            $movDetalle->justificacion = $justificacion;
            $movDetalle->costoinventario = $partida['importe'];
            $movDetalle->idsucursal_local = $almacen->idsucursal;
            $movDetalle->cantidad = $partida['cantidad'];
            $movDetalle->folio_lote_entrada = 0;
            $movDetalle->lote_entrada = 0;
            $movDetalle->idalmacen = $almacen->idwarehouse;
            $movDetalle->existencia = 0;
            $movDetalle->fisico = 0;
            $movDetalle->faltante = 0;
            $movDetalle->excedente = 0;
            $movDetalle->idsucursal_web_origen = 0;
            $movDetalle->idsucursal_web_destino = 0;
            $movDetalle->origen_registro = 'WEB';
            $movDetalle->descargado = 0;
            $movDetalle->idconceptoinventario = $conceptoInv->idinventoryconcept;

            $movDetalle->idfamilia = $producto->idfamilia;
            $movDetalle->idmarca = $producto->idmarca;
            $movDetalle->fecha_hora = $actualDate;
            $movDetalle->nombre_corto_producto = $producto->nombrecorto;
            $movDetalle->clave_familia = $producto->familia->clave ?? null;
            $movDetalle->clave_marca = $producto->marca->clave ?? null;

            $movDetalle->save();
        }
    }

    /**
     * Registra un traspaso de inventario entre dos almacenes/sucursales.
     * Genera DOS movimientos: SALIDA en origen y ENTRADA en destino.
     * 
     * @param array $data Datos del traspaso
     * @return array Retorna IDs de ambos movimientos y mensaje de éxito
     * @throws ValidationException
     */
    public function registrarTraspaso(array $data): array
    {
        Log::info('=== INICIANDO TRASPASO ===', [
            'almacen_origen_id' => $data['almacen_origen_id'] ?? 'no definido',
            'almacen_destino_id' => $data['almacen_destino_id'] ?? 'no definido',
            'partidas_count' => isset($data['partidas']) ? count($data['partidas']) : 0,
            'importe_total' => $data['importe_total'] ?? 0
        ]);
        
        $this->validateWith(\App\Http\Requests\Inventarios\RegistrarTraspasoRequest::class, $data);
        
        Log::info('Validación pasada correctamente');
        
        $partidas = $data['partidas'];
        $almacenOrigen = \App\Models\Almacen::with('sucursal')->findOrFail($data['almacen_origen_id']);
        $almacenDestino = \App\Models\Almacen::with('sucursal')->findOrFail($data['almacen_destino_id']);
        $proveedor = null;
        
        if (isset($data['proveedor_id']) && $data['proveedor_id']) {
            $proveedor = \App\Models\Proveedor::findOrFail($data['proveedor_id']);
        }
        
        $costoInventario = $data['importe_total'];
        $fecha = $data['fecha_movimiento'];
        $actualDate = Carbon::now();
        $nombreProveedor = $proveedor->nombre ?? '';
        
        // Validar que las sucursales sean diferentes
        if ($almacenOrigen->idsucursal === $almacenDestino->idsucursal) {
            throw ValidationException::withMessages([
                'almacen_destino_id' => 'El traspaso debe realizarse entre sucursales diferentes.'
            ]);
        }

        // Obtener usuario para sucursal origen
        Log::info('Buscando usuario para sucursal origen', ['idsucursal' => $almacenOrigen->idsucursal]);
        $userOrigen = Usuario::where('idsucursal', $almacenOrigen->idsucursal)
            ->first();
            
        if (!$userOrigen) {
            Log::error('No se encontró usuario para sucursal origen', ['idsucursal' => $almacenOrigen->idsucursal]);
            throw ValidationException::withMessages([
                'almacen_origen_id' => 'No se encontró un usuario válido para registrar movimientos en la sucursal origen.'
            ]);
        }
        
        Log::info('Usuario origen encontrado', ['id' => $userOrigen->iduser, 'nombre' => $userOrigen->nombre]);

        // Obtener usuario para sucursal destino
        Log::info('Buscando usuario para sucursal destino', ['idsucursal' => $almacenDestino->idsucursal]);
        $userDestino = Usuario::where('idsucursal', $almacenDestino->idsucursal)
            ->first();
            
        if (!$userDestino) {
            Log::error('No se encontró usuario para sucursal destino', ['idsucursal' => $almacenDestino->idsucursal]);
            throw ValidationException::withMessages([
                'almacen_destino_id' => 'No se encontró un usuario válido para registrar movimientos en la sucursal destino.'
            ]);
        }
        
        Log::info('Usuario destino encontrado', ['id' => $userDestino->iduser, 'nombre' => $userDestino->nombre]);

        // Buscar conceptos de SALIDA y ENTRADA
        $conceptoSalida = \App\Models\Concepto::where('conceptoinventario', 'SALIDA POR INVENTARIOS')->first();
        $conceptoEntrada = \App\Models\Concepto::where('conceptoinventario', 'ENTRADA POR INVENTARIOS')->first();

        if (!$conceptoSalida) {
            throw ValidationException::withMessages([
                'error' => 'No se encontró el concepto "SALIDA POR INVENTARIOS". Contacte al administrador.'
            ]);
        }

        if (!$conceptoEntrada) {
            throw ValidationException::withMessages([
                'error' => 'No se encontró el concepto "ENTRADA POR INVENTARIOS". Contacte al administrador.'
            ]);
        }

        // ============================
        // PASO 1: REGISTRAR SALIDA EN SUCURSAL ORIGEN
        // ============================
        Log::info('Creando movimiento de SALIDA', [
            'idsucursal' => $almacenOrigen->idsucursal,
            'idalmacen' => $almacenOrigen->idwarehouse,
            'idusuario' => $userOrigen->iduser,
            'costo' => $costoInventario
        ]);
        
        $movInvGeneralOrigen = MovimientoInventarioGeneral::create([
            'idsucursal'             => $almacenOrigen->idsucursal,
            'idinventariofisicogral' => 0,
            'fecha'                  => $fecha,
            'idalmacen'              => $almacenOrigen->idwarehouse,
            'estatus'                => Status::CERRADA,
            'idconceptoinventario'   => $conceptoSalida->idinventoryconcept,
            'idusuario'              => $userOrigen->iduser,
            'costoinventario'        => $costoInventario,
            'idalmacendestino'       => $almacenDestino->idwarehouse,
            'idorigeninventario'     => 3, // INVENTARIO FISICO
            'comentarios'            => '(TIPO MOVIMIENTO=TRASPASO) - SALIDA POR TRASPASO A ' . 
                                       ($almacenDestino->sucursal->razon_social ?? 'SUCURSAL DESTINO'),
            'horamovimiento'         => $actualDate->format('H:i:s'),
            'nombre_proveedor'       => $nombreProveedor,
            'idturno'                => 0,
            'fechahora_registro'     => $actualDate,
            'usuario'                => $userOrigen->nombre . ' - WEB',
            'from_host'              => 'SERVIDOR WEB',
            'origen_movimiento'      => '(TRASPASO) SALIDA POR INVENTARIOS',
            'tipo_movimiento'        => 'SALIDA',
            'signo'                  => -1, // NEGATIVO (resta existencias)
            'origen_registro'        => 'WEB',
            'descargado'             => 0,
            'idproveedor'            => $proveedor->idprovider ?? null,
            'idmovinventariogral_desk' => 0,
        ]);

        if (!$movInvGeneralOrigen) {
            Log::error('Error al crear movimiento de SALIDA');
            throw ValidationException::withMessages([
                'error' => 'Error al registrar el movimiento de SALIDA en sucursal origen.'
            ]);
        }
        
        Log::info('Movimiento de SALIDA creado exitosamente', ['id' => $movInvGeneralOrigen->idmovinventariogral]);

        // Registrar detalles de SALIDA
        Log::info('Registrando detalles de SALIDA');
        $this->registrarMovimientosDetalleTraspaso(
            $partidas, 
            $almacenOrigen, 
            $movInvGeneralOrigen, 
            $conceptoSalida, 
            $actualDate,
            'SALIDA POR TRASPASO',
            $almacenOrigen->idsucursal,
            $almacenDestino->idsucursal
        );
        Log::info('Detalles de SALIDA registrados exitosamente');

        // ============================
        // PASO 2: REGISTRAR ENTRADA EN SUCURSAL DESTINO
        // ============================
        Log::info('Creando movimiento de ENTRADA', [
            'idsucursal' => $almacenDestino->idsucursal,
            'idalmacen' => $almacenDestino->idwarehouse,
            'idusuario' => $userDestino->iduser
        ]);
        $movInvGeneralDestino = MovimientoInventarioGeneral::create([
            'idsucursal'             => $almacenDestino->idsucursal,
            'idinventariofisicogral' => 0,
            'fecha'                  => $fecha,
            'idalmacen'              => $almacenDestino->idwarehouse,
            'estatus'                => Status::CERRADA,
            'idconceptoinventario'   => $conceptoEntrada->idinventoryconcept,
            'idusuario'              => $userDestino->iduser,
            'costoinventario'        => $costoInventario,
            'idalmacendestino'       => 0, // Destino es este mismo almacén
            'idorigeninventario'     => 3, // INVENTARIO FISICO
            'comentarios'            => '(TIPO MOVIMIENTO=TRASPASO) - ENTRADA POR TRASPASO DESDE ' . 
                                       ($almacenOrigen->sucursal->razon_social ?? 'SUCURSAL ORIGEN'),
            'horamovimiento'         => $actualDate->format('H:i:s'),
            'nombre_proveedor'       => $nombreProveedor,
            'idturno'                => 0,
            'fechahora_registro'     => $actualDate,
            'usuario'                => $userDestino->nombre . ' - WEB',
            'from_host'              => 'SERVIDOR WEB',
            'origen_movimiento'      => '(TRASPASO) ENTRADA POR INVENTARIOS',
            'tipo_movimiento'        => 'ENTRADA',
            'signo'                  => 1, // POSITIVO (suma existencias)
            'origen_registro'        => 'WEB',
            'descargado'             => 0,
            'idproveedor'            => $proveedor->idprovider ?? null,
            'idmovinventariogral_desk' => 0,
        ]);

        if (!$movInvGeneralDestino) {
            Log::error('Error al crear movimiento de ENTRADA');
            throw ValidationException::withMessages([
                'error' => 'Error al registrar el movimiento de ENTRADA en sucursal destino.'
            ]);
        }
        
        Log::info('Movimiento de ENTRADA creado exitosamente', ['id' => $movInvGeneralDestino->idmovinventariogral]);

        // Registrar detalles de ENTRADA
        Log::info('Registrando detalles de ENTRADA');
        $this->registrarMovimientosDetalleTraspaso(
            $partidas, 
            $almacenDestino, 
            $movInvGeneralDestino, 
            $conceptoEntrada, 
            $actualDate,
            'ENTRADA POR TRASPASO',
            $almacenOrigen->idsucursal,
            $almacenDestino->idsucursal
        );
        Log::info('Detalles de ENTRADA registrados exitosamente');

        Log::info('=== TRASPASO COMPLETADO EXITOSAMENTE ===', [
            'id_salida' => $movInvGeneralOrigen->idmovinventariogral,
            'id_entrada' => $movInvGeneralDestino->idmovinventariogral
        ]);

        return [
            'status' => 1,
            'message' => 'El traspaso se registró exitosamente. Se generaron ' .
                        'SALIDA en ' . ($almacenOrigen->sucursal->razon_social ?? 'origen') . 
                        ' y ENTRADA en ' . ($almacenDestino->sucursal->razon_social ?? 'destino') . '.',
            'idmovinventariogral_origen' => $movInvGeneralOrigen->idmovinventariogral,
            'idmovinventariogral_destino' => $movInvGeneralDestino->idmovinventariogral,
        ];
    }

    /**
     * Registra los detalles de movimientos para traspasos.
     * Similar a registrarMovimientosDetalle pero con campos específicos de traspaso.
     */
    private function registrarMovimientosDetalleTraspaso(
        array $partidas, 
        $almacen, 
        $movInvGeneral, 
        $conceptoInv, 
        $actualDate,
        string $justificacionBase,
        int $sucursalOrigenId,
        int $sucursalDestinoId
    ): void {
        Log::info('Iniciando registro de detalles', [
            'partidas_count' => count($partidas),
            'movimiento_id' => $movInvGeneral->idmovinventariogral
        ]);
        
        foreach ($partidas as $index => $partida) {
            Log::info("Procesando partida {$index}", ['producto_id' => $partida['producto_id'] ?? 'sin id']);
            
            $justificacion = $partida['justificacion'] ?? '';
            
            if (StringUtils::isEmpty($justificacion)) {
                $justificacion = $justificacionBase;
            } else {
                $justificacion = $justificacionBase . ' - ' . $justificacion;
            }

            $producto = Producto::with(['familia', 'marca'])->find($partida['producto_id']);
            
            if (!$producto) {
                Log::error('Producto no encontrado', ['producto_id' => $partida['producto_id'] ?? 'sin datos']);
                throw ValidationException::withMessages([
                    'error' => 'No se encontró el producto con ID ' . ($partida['producto_id'] ?? 'sin datos')
                ]);
            }
            
            Log::info('Producto encontrado', ['id' => $producto->idproduct, 'nombre' => $producto->nombrecorto]);

            try {
                $movDetalle = new MovimientoInventarioDetalle();
                $movDetalle->idsucursal = $almacen->idsucursal;
                $movDetalle->idinventariofisicodet = 0;
                $movDetalle->idinventariofisicogral = $movInvGeneral->idmovinventariogral;
                $movDetalle->idproducto = $producto->idproduct;
                $movDetalle->justificacion = $justificacion;
                $movDetalle->costoinventario = $partida['importe'];
                $movDetalle->idsucursal_local = $almacen->idsucursal;
                $movDetalle->cantidad = $partida['cantidad'];
                $movDetalle->folio_lote_entrada = 0;
                $movDetalle->lote_entrada = $partida['lote_origen'] ?? '';
                $movDetalle->idalmacen = $almacen->idwarehouse;
                $movDetalle->existencia = 0;
                $movDetalle->fisico = 0;
                $movDetalle->faltante = 0;
                $movDetalle->excedente = 0;
                $movDetalle->idsucursal_web_origen = $sucursalOrigenId;
                $movDetalle->idsucursal_web_destino = $sucursalDestinoId;
                $movDetalle->origen_registro = 'WEB';
                $movDetalle->descargado = 0;
                $movDetalle->idconceptoinventario = $conceptoInv->idinventoryconcept;

                $movDetalle->idfamilia = $producto->idfamilia;
                $movDetalle->idmarca = $producto->idmarca;
                $movDetalle->fecha_hora = $actualDate;
                $movDetalle->nombre_corto_producto = $producto->nombrecorto;
                $movDetalle->clave_familia = $producto->familia->clave ?? null;
                $movDetalle->clave_marca = $producto->marca->clave ?? null;

                Log::info('Intentando guardar detalle', ['partida' => $index]);
                $movDetalle->save();
                Log::info('Detalle guardado exitosamente', ['id' => $movDetalle->idinventariofisicodet, 'partida' => $index]);
            } catch (\Exception $e) {
                Log::error('Error al guardar detalle', [
                    'partida' => $index,
                    'error' => $e->getMessage(),
                    'trace' => $e->getTraceAsString()
                ]);
                throw $e;
            }
        }
        
        Log::info('Todos los detalles guardados exitosamente');
    }
}
