<?php

namespace App\Repositories\PuntoDeVenta;

use App\Exceptions\PuntoDeVenta\ClienteNoEncontradoException;
use App\Exceptions\PuntoDeVenta\SucursalNoEncontradaException;
use App\Livewire\PuntoDeVenta\PuntoVentas\PuntoVentasBaseLogic;
use App\Models\Cliente;
use App\Models\Sucursal;
use App\Models\Producto;
use App\Models\VentasGeneral;
use App\Models\VentasDetalle;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\Log;

class PuntoDeVentaRepository implements PuntoDeVentaInterface
{
    public function crearVenta(array $data)
    {
        $idclient = $data['idcliente'] ?? $data['idclient'] ?? null;
        $sucursalid = $data['sucursalid'] ?? null;
        $productos = $data['productosagregados'] ?? $data['productos'] ?? [];

        $empresaId = 42;
        try {
            if (function_exists('tenancy')) {
                if (! tenancy()->initialized || tenant('id') !== $empresaId) {
                    \Illuminate\Support\Facades\Log::info('[PuntoDeVenta] Inicializando tenancy forzado al tenant ' . $empresaId);
                    tenancy()->initialize($empresaId);
                }
            }
        } catch (\Throwable $e) {
            \Illuminate\Support\Facades\Log::warning('[PuntoDeVenta] No se pudo inicializar tenancy forzado al inicio: ' . $e->getMessage());
        }

        if (empty($idclient) || empty($sucursalid)) {
            throw new \InvalidArgumentException('idclient y sucursalid son requeridos');
        }
        if (!is_array($productos) || count($productos) === 0) {
            throw new \InvalidArgumentException('Debe enviar al menos un producto');
        }

        $cliente = Cliente::query()
            ->where(function ($q) use ($idclient) {
                $q->where('idcliente', $idclient)
                    ->orWhere('idclient', $idclient);
            })
            ->first();
        if (!$cliente) {
            throw new ClienteNoEncontradoException('Cliente no encontrado');
        }

        $sucursal = $this->buscarSucursal($sucursalid);

        $totales = PuntoVentasBaseLogic::calcularTotalesSimplificados($productos);
        $totalFinal = $totales['totalfinal'] ?? 0.0;
        $agregados = $this->sumarTotalesProductos($productos);

        $folio = $data['folio'] ?? ('PV-' . date('Ymd-His'));

        $cabecera = null;
        DB::transaction(function () use ($data, $cliente, $sucursal, $productos, $agregados, $totalFinal, $empresaId, &$cabecera) {
            $cabecera = $this->crearCabeceraVenta($data, $cliente, $sucursal, $agregados, $totalFinal, $empresaId);
            foreach ($productos as $p) {
                DB::table('ventasdetalle')->insert($this->mapDetalleRow($p, $cabecera, $sucursal, $empresaId));
            }

            $pagos = $data['pagos'] ?? [];
            $tieneCredito30 = false;
            if (is_array($pagos) && count($pagos) > 0) {
                foreach ($pagos as $pago) {
                    $idtipopago = $pago['idtipopago'] ?? $pago['id'] ?? null;
                    $importe = $pago['importe'] ?? $pago['monto'] ?? 0;
                    $referencia = $pago['referencia'] ?? null;

                    $tablePagos = 'vtas_pagosdeventas';
                    if (Schema::hasTable($tablePagos)) {
                        $insert = [];
                        $set = $this->makeColumnSetter($tablePagos);
                        $set($insert, 'idventageneral', $cabecera->idventagral);
                        $set($insert, 'idtipopago', $idtipopago);
                        $set($insert, 'tipopago', $pago['tipopago'] ?? null);
                        $set($insert, 'importe', $importe);
                        $set($insert, 'referencia', $referencia);
                        $set($insert, 'fechahora', now());
                        $set($insert, 'created_at', now());
                        $set($insert, 'updated_at', now());

                        if (!empty($insert)) {
                            DB::table($tablePagos)->insert($insert);
                        }
                    }

                    if ($idtipopago == 5 || str_contains(strtoupper($pago['tipopago'] ?? ''), 'CREDITO')) {
                        $tieneCredito30 = true;
                    }
                }
            } else {
                $tablePagos = 'vtas_pagosdeventas';
                if (Schema::hasTable($tablePagos)) {
                    $insert = [];
                    $set = $this->makeColumnSetter($tablePagos);
                    $set($insert, 'idventageneral', $cabecera->idventagral);
                    $set($insert, 'idtipopago', $data['idtipopago'] ?? 1);
                    $set($insert, 'tipopago', $data['tipopago'] ?? 'EFECTIVO');
                    $set($insert, 'importe', $data['totalentregado'] ?? $totalFinal);
                    $set($insert, 'referencia', null);
                    $set($insert, 'fechahora', now());
                    $set($insert, 'created_at', now());
                    $set($insert, 'updated_at', now());

                    if (!empty($insert)) {
                        DB::table($tablePagos)->insert($insert);
                    }
                }
                $defaultIdTipoPago = $data['idtipopago'] ?? null;
                $defaultTipoPagoText = strtoupper($data['tipopago'] ?? '');
                if ($defaultIdTipoPago == 5 || str_contains($defaultTipoPagoText, 'CREDITO')) {
                    $tieneCredito30 = true;
                }
            }


            if ($tieneCredito30) {
                $debug = [
                    'tieneCredito30' => $tieneCredito30,
                    'pagos' => $pagos,
                    'schema_has_vtas_cxcgeneral' => Schema::hasTable('vtas_cxcgeneral'),
                    'schema_has_vtas_cobranza' => Schema::hasTable('vtas_cobranza'),
                    'connection_database' => DB::connection()->getDatabaseName(),
                    'tenant_id' => function_exists('tenant') ? tenant('id') : null,
                ];
                Log::info('[PuntoDeVenta][crearVenta] crédito 30 debug', $debug);

                $montoCredito = 0.0;
                if (is_array($pagos)) {
                    foreach ($pagos as $p) {
                        $pid = $p['idtipopago'] ?? $p['id'] ?? null;
                        $nom = strtoupper($p['tipopago'] ?? '');
                        if ($pid == 5 || str_contains($nom, 'CREDITO')) {
                            $montoCredito += floatval($p['importe'] ?? 0);
                        }
                    }
                }
                if ($montoCredito <= 0) {
                    $montoCredito = floatval($cabecera->totalapagar ?? $totalFinal);
                }

                $tableCxc = $this->resolveTenantTableName(['vtas_cxcgeneral', 'cxcgeneral', 'vtas_cxc_general', 'CXCGENERAL', 'CXC_GENERAL']);
                if ($tableCxc) {
                    try {
                        $vencimiento = now()->addDays(30)->toDateString();
                        $insertCxc = [];
                        $totalDoc = $montoCredito;
                        $cols = [];
                        try { $cols = Schema::getColumnListing($tableCxc); } catch (\Throwable $e) { $cols = []; }
                        $colMap = [];
                        foreach ($cols as $c) { $colMap[strtolower($c)] = $c; }
                        $set = function(string $nameLower, $value) use (&$insertCxc, $colMap) {
                            $key = strtolower($nameLower);
                            if (isset($colMap[$key])) { $insertCxc[$colMap[$key]] = $value; }
                        };
                        $set('idventageneral', $cabecera->idventagral);
                        $set('idcliente', $cliente->idclient);
                        $set('fechahora_documento', now());
                        $set('fechagenera', now()->toDateString());
                        $set('fechavencimiento', $vencimiento);
                        $set('plazo_dias', 30);
                        $set('estado', 'PENDIENTE');
                        $set('importe_documento', $totalDoc);
                        $set('importe', $totalDoc);
                        $set('total_abonado', 0);
                        $set('saldo_documento', $totalDoc);
                        $set('saldo', $totalDoc);
                        $set('credito_cobrado_mismo_turno', 0);
                        $set('observaciones', 'Creado desde POS (Crédito 30 días)');
                        if (!empty($data['idturno'] ?? null)) { $set('idturno', $data['idturno']); }
                        $set('idusuario', Auth::id() ?? 0);
                        $set('usuario', optional(Auth::user())->name ?? 'api');
                        $set('origen_regisro', 'POS');
                        $set('idenweb', 0);
                        $set('created_at', now());
                        $set('updated_at', now());

                        if (!empty($insertCxc)) {
                            DB::table($tableCxc)->insert($insertCxc);
                            Log::info('[PuntoDeVenta][crearVenta] vtas_cxcgeneral insert OK', ['table' => $tableCxc, 'idventageneral' => $cabecera->idventagral]);
                        } else {
                            Log::warning('[PuntoDeVenta][crearVenta] vtas_cxcgeneral - no hay columnas válidas para insertar', ['table' => $tableCxc]);
                        }
                    } catch (\Throwable $e) {
                        Log::error('[PuntoDeVenta][crearVenta] Error insert vtas_cxcgeneral: ' . $e->getMessage(), ['exception' => $e]);
                    }
                } else {
                    Log::warning('[PuntoDeVenta][crearVenta] vtas_cxcgeneral no existe en esta conexión (busqueda variantes)', ['database' => DB::connection()->getDatabaseName()]);
                }

                $tableCob = $this->resolveTenantTableName(['vtas_cobranza', 'cobranza', 'VTAS_COBRANZA']);
                if ($tableCob) {
                    try {
                        $insertCob = [];
                        $cols2 = [];
                        try { $cols2 = Schema::getColumnListing($tableCob); } catch (\Throwable $e) { $cols2 = []; }
                        $colMap2 = [];
                        foreach ($cols2 as $c) { $colMap2[strtolower($c)] = $c; }
                        $set2 = function(string $nameLower, $value) use (&$insertCob, $colMap2) {
                            $key = strtolower($nameLower);
                            if (isset($colMap2[$key])) { $insertCob[$colMap2[$key]] = $value; }
                        };

                        $set2('idventageneral', $cabecera->idventagral);
                        $set2('idcliente', $cliente->idclient);
                        $set2('monto', $montoCredito);
                        $set2('fecha_programada', $vencimiento ?? now()->addDays(30)->toDateString());
                        $set2('estado', 'PROGRAMADA');
                        $set2('created_at', now());
                        $set2('updated_at', now());

                        if (!empty($insertCob)) {
                            DB::table($tableCob)->insert($insertCob);
                            Log::info('[PuntoDeVenta][crearVenta] vtas_cobranza insert OK', ['table' => $tableCob, 'idventageneral' => $cabecera->idventagral]);
                        } else {
                            Log::warning('[PuntoDeVenta][crearVenta] vtas_cobranza - no hay columnas válidas para insertar', ['table' => $tableCob]);
                        }
                    } catch (\Throwable $e) {
                        Log::error('[PuntoDeVenta][crearVenta] Error insert vtas_cobranza: ' . $e->getMessage(), ['exception' => $e]);
                    }
                } else {
                    Log::warning('[PuntoDeVenta][crearVenta] vtas_cobranza no existe en esta conexión (busqueda variantes)', ['database' => DB::connection()->getDatabaseName()]);
                }

                $montoNoCredito = 0.0;
                if (is_array($pagos)) {
                    foreach ($pagos as $p) {
                        $pid = $p['idtipopago'] ?? $p['id'] ?? null;
                        $nom = strtoupper($p['tipopago'] ?? '');
                        $imp = floatval($p['importe'] ?? 0);
                        if (!($pid == 5 || str_contains($nom, 'CREDITO'))) {
                            $montoNoCredito += $imp;
                        }
                    }
                }
                $cabecera->total_xcobrar = $montoCredito;
                $cabecera->total_cobrado = $montoNoCredito;
                $cabecera->save();
            }
        });

        $pdf_base64 = null;

        return [
            'folio' => $folio,
            'filename' => 'venta_' . date('Ymd_His') . '.pdf',
            'pdf_base64' => $pdf_base64,
            'totales' => $totales,
        ];
    }

    private function makeColumnSetter(string $table): callable
    {
        $cols = [];
        try {
            $cols = Schema::getColumnListing($table);
        } catch (\Throwable $e) {
            $cols = [];
        }
        $map = [];
        foreach ($cols as $c) {
            $map[strtolower($c)] = $c;
        }
        return function (array &$dest, string $nameLower, $value) use ($map) {
            $key = strtolower($nameLower);
            if (isset($map[$key])) {
                $dest[$map[$key]] = $value;
            }
        };
    }

    /**
     * Resolve a tenant table name from a list of likely variants.
     * Returns the first variant that exists in the current connection's schema,
     * otherwise returns the first variant (best-effort) so calling code may still attempt insert.
     *
     * @param array $variants
     * @return string
     */
    private function resolveTenantTableName(array $variants): string
    {
        foreach ($variants as $v) {
            try {
                if (Schema::hasTable($v)) {
                    return $v;
                }
            } catch (\Exception $e) {
                Log::warning("[PuntoDeVenta][resolveTenantTableName] schema check failed for {$v}: " . $e->getMessage());
            }
        }

        return $variants[0] ?? '';
    }

    public function obtenerVentas(array $filters)
    {
        $query = VentasGeneral::query();
        if (!empty($filters['idcliente'])) {
            $query->where('idcliente', $filters['idcliente']);
        }
        if (!empty($filters['fecha_desde'])) {
            $query->whereDate('fechaventa', '>=', $filters['fecha_desde']);
        }
        if (!empty($filters['fecha_hasta'])) {
            $query->whereDate('fechaventa', '<=', $filters['fecha_hasta']);
        }
        $data = $query->orderByDesc('idventagral')->limit(50)->get();
        return [
            'data' => $data,
            'filters' => $filters,
        ];
    }

    public function obtenerVentaPorId(int $id)
    {
        $query = VentasGeneral::query();
        return $query->find($id);
    }

    public function obtenerProductos(array $filters)
    {
        $query = Producto::query();
        if (!empty($filters['id'])) {
            $query->where('idproduct', $filters['id']);
        }
        return $query->first();
    }

    private function sumarTotalesProductos(array $productos): array
    {
        $scale = 8;
        $sumSubtotal = '0';
        $sumIva = '0';
        $sumRetIva = '0';
        $sumRetIsr = '0';
        $sumDescuento = '0';
        foreach ($productos as $p) {
            $sumSubtotal = bcadd($sumSubtotal, (string) ($p['subtotal'] ?? $p['total'] ?? 0), $scale);
            $sumIva = bcadd($sumIva, (string) ($p['iva_importe'] ?? $p['iva_trasladado'] ?? 0), $scale);
            $sumRetIva = bcadd($sumRetIva, (string) ($p['retencion_iva_importe'] ?? $p['iva_retenido'] ?? 0), $scale);
            $sumRetIsr = bcadd($sumRetIsr, (string) ($p['retencion_isr_importe'] ?? $p['isr_retenido'] ?? 0), $scale);
            $sumDescuento = bcadd($sumDescuento, (string) ($p['descuento_total'] ?? $p['descuento'] ?? 0), $scale);
        }
        return [
            'subtotal' => $sumSubtotal,
            'iva' => $sumIva,
            'ret_iva' => $sumRetIva,
            'ret_isr' => $sumRetIsr,
            'descuento' => $sumDescuento,
        ];
    }

    private function crearCabeceraVenta(array $data, $cliente, $sucursal, array $agregados, $totalFinal)
    {
        $payload = [
            'idsucursal' => $sucursal->idsucursal ?? $sucursal->idbranchoffice ?? null,
            'idventageneral' => 0,
            'idstatus' => 1,
            'fechaventa' => date('Y-m-d'),
            'horaventa' => date('H:i:s'),
            'fechahora' => now(),
            'idcliente' => $cliente->idclient,
            'cliente' => $cliente->nombre,
            'idtipodocumento' => $data['idtipodocumento'] ?? 3, // 3 = NOTA DE VENTA
            'documento' => 'NOTA DE VENTA',
            'folio' => $this->extraerNumeroFolio($data['folio'] ?? ('PV-' . date('YmdHis'))),
            'pagado' => (int) ($data['pagado'] ?? 1),
            'idtipopago' => $data['idtipopago'] ?? 1,
            'tipopago' => $data['tipopago'] ?? 'EFECTIVO',
            'idusuario' => Auth::id() ?? 0,
            'importe' => $agregados['subtotal'],
            'descuentox100' => $data['descuentox100'] ?? 0,
            'descuento' => $agregados['descuento'],
            'subtotal' => $agregados['subtotal'],
            'iva' => $agregados['iva'],
            'total' => (string) $totalFinal,
            'totalentregado' => $data['totalentregado'] ?? $totalFinal,
            'cambioaentregar' => $data['cambioaentregar'] ?? 0,
            'comision' => $data['comision'] ?? -1,
            'observaciones1' => $data['observaciones1'] ?? '',
            'idturno' => $data['idturno'] ?? null,
            'usuario' => optional(Auth::user())->name ?? 'api',
            'costototal' => $data['costototal'] ?? 0,
            'utilidadtotal' => $data['utilidadtotal'] ?? 0,
            'from_host' => request()->ip(),
            'origen_movimiento' => 'POS',
            'descargado' => 0,
            'origen_registro' => 'WEB',
            'deletedofdesk' => 0,
        ];

        return VentasGeneral::create($payload);
    }
    
    private function extraerNumeroFolio($folio)
    {
        if (preg_match('/\d+$/', $folio, $matches)) {
            return (int) $matches[0];
        }
        return (int) str_replace(['PV-', '-'], '', $folio);
    }

    private function buscarSucursal($sucursalid)
    {
        $sucursal = Sucursal::query()
            ->where(function ($q) use ($sucursalid) {
                $q->where('idbranchoffice', $sucursalid)
                    ->orWhere('idsucursal', $sucursalid);
            })
            ->first();

        if ($sucursal) {
            return $sucursal;
        }

        $count = Sucursal::query()->count();
        if ($count === 1) {
            return Sucursal::query()->first();
        }

        throw new SucursalNoEncontradaException('Sucursal no encontrada');
    }

    private function mapDetalleRow(array $p, $cabecera, $sucursal): array
    {
        $row = [
            'idventadetalle_desk' => null,
            'idventageneral' => $cabecera->idventagral,
            'idsucursal' => $sucursal->idsucursal ?? $sucursal->idbranchoffice ?? null,
            'idproducto' => $p['producto_id'] ?? $p['idproducto'] ?? null,
            'idfamilia' => $p['idfamilia'] ?? null,
            'idmarca' => $p['idmarca'] ?? null,
            'clave' => $p['clave'] ?? '',
            'descripcion' => $p['descripcion'] ?? '',
            'cantidad' => $p['cantidad'] ?? 1,
            'precio' => $p['precio_unitario'] ?? 0,
            'importe' => $p['importe'] ?? $p['total'] ?? 0,
            'fechahora' => now(),
            'descuentox100' => $p['descuento1_porcentaje'] ?? 0,
            'descuento' => $p['descuento_total'] ?? ($p['descuento'] ?? 0),
            'subtotal' => $p['subtotal'] ?? $p['total'] ?? 0,
            'iva' => $p['iva_importe'] ?? ($p['iva_trasladado'] ?? 0),
            'total' => $p['total'] ?? 0,
            'idalmacen' => $p['idalmacen'] ?? 1,
            'costopromedio' => $p['costopromedio'] ?? 0,
            'costo_venta' => $p['costo_venta'] ?? 0,
            'utilidad' => $p['utilidad'] ?? 0,
            'origen_registro' => 1,
            'clave_marca' => $p['clave_marca'] ?? null,
            'clave_familia' => $p['clave_familia'] ?? null,
            'idusuario' => Auth::id() ?? null,
            'idusuario_desk' => null,
            'idturno' => $cabecera->idturno ?? null,
            'idturno_desk' => null,
            'vtagral_idstatus' => $cabecera->idstatus ?? 1,
            'idtipodocumento' => $cabecera->idtipodocumento ?? 3,
            'folio' => $cabecera->folio ?? null,
        ];

        return $row;
    }
}

