import { Chart, registerables } from 'chart.js';

Chart.register(...registerables);

/**
 * @param {string} canvasId - ID del elemento canvas
 * @param {Array} labels - Etiquetas del eje X
 * @param {Array} data - Datos numéricos
 * @param {Object} options - Opciones adicionales (opcional)
 * @returns {Chart|null} - Instancia de Chart.js o null si hay error
 */
export function crearGraficaBarras(canvasId, labels, data, options = {}) {
    const ctx = document.getElementById(canvasId);
    
    if (!ctx) {
        console.error(`Canvas con ID "${canvasId}" no encontrado`);
        return null;
    }

    const defaultOptions = {
        responsive: true,
        maintainAspectRatio: true,
        plugins: {
            legend: {
                display: true,
                position: 'top',
            },
            title: {
                display: true,
                text: 'Gráfica de Barras',
                font: {
                    size: 16,
                    weight: 'bold'
                }
            },
            tooltip: {
                enabled: true,
                backgroundColor: 'rgba(0, 0, 0, 0.8)',
                titleColor: '#fff',
                bodyColor: '#fff',
                callbacks: {
                    label: function(context) {
                        let label = context.dataset.label || '';
                        if (label) {
                            label += ': ';
                        }
                        if (context.parsed.y !== null) {
                            label += '$' + context.parsed.y.toLocaleString('es-MX', {
                                minimumFractionDigits: 2,
                                maximumFractionDigits: 2
                            });
                        }
                        return label;
                    }
                }
            }
        },
        scales: {
            y: {
                beginAtZero: true,
                ticks: {
                    callback: function(value) {
                        return '$' + value.toLocaleString('es-MX');
                    }
                },
                grid: {
                    color: 'rgba(0, 0, 0, 0.05)'
                }
            },
            x: {
                grid: {
                    display: false
                }
            }
        }
    };

    const finalOptions = { ...defaultOptions, ...options };

    const config = {
        type: 'bar',
        data: {
            labels: labels,
            datasets: [{
                label: 'Datos',
                data: data,
                backgroundColor: 'rgba(0, 166, 90, 0.8)', // Verde
                borderColor: 'rgba(0, 166, 90, 1)',
                borderWidth: 2,
                borderRadius: 5,
                borderSkipped: false,
            }]
        },
        options: finalOptions
    };

    return new Chart(ctx, config);
}

/**
 * @param {Chart} chart - Instancia de Chart.js
 * @param {Array} labels - Nuevas etiquetas
 * @param {Array} data - Nuevos datos
 */
export function actualizarGrafica(chart, labels, data) {
    if (!chart) {
        console.error('La gráfica no existe');
        return;
    }

    chart.data.labels = labels;
    chart.data.datasets[0].data = data;
    chart.update();
}

/**
 * @param {string} canvasId - ID del elemento canvas
 * @param {Array} labels - Etiquetas del eje X
 * @param {Array} data - Datos numéricos
 * @param {Object} options - Opciones adicionales (opcional)
 * @returns {Chart|null} - Instancia de Chart.js o null si hay error
 */
export function crearGraficaLineas(canvasId, labels, data, options = {}) {
    const ctx = document.getElementById(canvasId);
    
    if (!ctx) {
        console.error(`Canvas con ID "${canvasId}" no encontrado`);
        return null;
    }

    const defaultOptions = {
        responsive: true,
        maintainAspectRatio: true,
        plugins: {
            legend: {
                display: true,
                position: 'top',
            },
            title: {
                display: true,
                text: 'Gráfica de Líneas',
                font: {
                    size: 16,
                    weight: 'bold'
                }
            }
        },
        scales: {
            y: {
                beginAtZero: true
            }
        }
    };

    const finalOptions = { ...defaultOptions, ...options };

    const config = {
        type: 'line',
        data: {
            labels: labels,
            datasets: [{
                label: 'Datos',
                data: data,
                borderColor: 'rgba(0, 166, 90, 1)',
                backgroundColor: 'rgba(0, 166, 90, 0.1)',
                tension: 0.4, // Suavizado de líneas
                fill: true
            }]
        },
        options: finalOptions
    };

    return new Chart(ctx, config);
}

/**
 * @param {string} canvasId - ID del elemento canvas
 * @param {Array} labels - Etiquetas
 * @param {Array} data - Datos numéricos
 * @param {string} type - Tipo: 'pie' o 'doughnut'
 * @param {Object} options - Opciones adicionales (opcional)
 * @returns {Chart|null} - Instancia de Chart.js o null si hay error
 */
export function crearGraficaCircular(canvasId, labels, data, type = 'pie', options = {}) {
    const ctx = document.getElementById(canvasId);
    
    if (!ctx) {
        console.error(`Canvas con ID "${canvasId}" no encontrado`);
        return null;
    }

    const colores = [
        'rgba(255, 99, 132, 0.8)',
        'rgba(54, 162, 235, 0.8)',
        'rgba(255, 206, 86, 0.8)',
        'rgba(75, 192, 192, 0.8)',
        'rgba(153, 102, 255, 0.8)',
        'rgba(255, 159, 64, 0.8)',
        'rgba(0, 166, 90, 0.8)',
    ];

    const defaultOptions = {
        responsive: true,
        maintainAspectRatio: true,
        plugins: {
            legend: {
                display: true,
                position: 'right',
            },
            title: {
                display: true,
                text: type === 'pie' ? 'Gráfica Circular' : 'Gráfica de Dona',
                font: {
                    size: 16,
                    weight: 'bold'
                }
            }
        }
    };

    const finalOptions = { ...defaultOptions, ...options };

    const config = {
        type: type,
        data: {
            labels: labels,
            datasets: [{
                label: 'Datos',
                data: data,
                backgroundColor: colores.slice(0, data.length),
                borderWidth: 2
            }]
        },
        options: finalOptions
    };

    return new Chart(ctx, config);
}

/**
 * @param {Chart} chart - Instancia de Chart.js
 */
export function destruirGrafica(chart) {
    if (chart) {
        chart.destroy();
    }
}

globalThis.crearGraficaBarras = crearGraficaBarras;
globalThis.crearGraficaLineas = crearGraficaLineas;
globalThis.crearGraficaCircular = crearGraficaCircular;
globalThis.actualizarGrafica = actualizarGrafica;
globalThis.destruirGrafica = destruirGrafica;
