<?php

use App\Models\CentralUser;
use App\Models\User;
use App\Models\Tenant;
use Illuminate\Auth\Events\Registered;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\Rules;
use Livewire\Attributes\Layout;
use Livewire\Volt\Component;

new #[Layout('components.layouts.auth')] class extends Component {
    public string $nombre = '';
    public string $email = '';
    public string $password = '';
    public string $password_confirmation = '';
    public string $tenant_name = '';

    /**
     * Handle an incoming registration request.
     */
    public function register(): void
    {
        $validated = $this->validate([
            'nombre' => ['required', 'string', 'max:255'],
            'email' => ['required', 'string', 'lowercase', 'email', 'max:255', 'unique:' . CentralUser::class],
            'password' => ['required', 'string', 'confirmed', Rules\Password::defaults()],
            'tenant_name' => ['required', 'string', 'alpha_dash', 'max:255', 'unique:tenants,id'],
        ]);

        $validated['password'] = Hash::make($validated['password']);

        $tenantName = $this->tenant_name;
        $newTenant = Tenant::create(['id' => $tenantName]);
        $newTenant->domains()->create(['domain' => "$tenantName." . env('APP_DOMAIN')]);
        $centralUser = CentralUser::create([
            'nombre' => $validated['nombre'],
            'email' => $validated['email'],
            'password' => $validated['password'],
            'tenant_id' => $tenantName,
        ]);
        event(new Registered(($user = $centralUser)));

        tenancy()->initialize($newTenant);

        User::create([
            'nombre' => $centralUser->nombre,
            'email' => $centralUser->email,
            'central_user_id' => $centralUser->id,
        ]);

        tenancy()->end();

        Auth::login($user);

        $this->redirectIntended(route('dashboard', absolute: false), navigate: true);
    }
}; ?>

<div class="flex flex-col gap-6">
    <x-auth-header :title="__('Create an account')" :description="__('Enter your details below to create your account')" />

    <!-- Session Status -->
    <x-auth-session-status class="text-center" :status="session('status')" />

    <form wire:submit="register" class="flex flex-col gap-6">
        <!-- Tenant Name -->
        <flux:field>
            <flux:label>{{ __('Subdomain name') }}</flux:label>
            <flux:input.group>
                <flux:input wire:model="tenant_name" type="text" required autofocus autocomplete="tenant_name"
                    :placeholder="__('subdomain-example')" />

                <flux:input.group.suffix>.{{ env('APP_DOMAIN') }}</flux:input.group.suffix>
            </flux:input.group>
            <flux:error name="tenant_name" />

            <flux:description>
                {{ __('This will be your subdomain. It must be unique and can only contain letters, numbers, dashes, and underscores.') }}
            </flux:description>
        </flux:field>

        <!-- Name -->
        <flux:input wire:model="nombre" :label="__('Name')" type="text" required autofocus autocomplete="nombre"
            :placeholder="__('Full name')" />

        <!-- Email Address -->
        <flux:input wire:model="email" :label="__('Email address')" type="email" required autocomplete="email"
            placeholder="email@example.com" />

        <!-- Password -->
        <flux:input wire:model="password" :label="__('Password')" type="password" required autocomplete="new-password"
            :placeholder="__('Password')" />

        <!-- Confirm Password -->
        <flux:input wire:model="password_confirmation" :label="__('Confirm password')" type="password" required
            autocomplete="new-password" :placeholder="__('Confirm password')" />

        <div class="flex items-center justify-end">
            <flux:button type="submit" variant="primary" class="w-full">
                {{ __('Create account') }}
            </flux:button>
        </div>
    </form>

    <div class="space-x-1 rtl:space-x-reverse text-center text-sm text-zinc-600 dark:text-zinc-400">
        {{ __('Already have an account?') }}
        <flux:link :href="route('login')" wire:navigate>{{ __('Log in') }}</flux:link>
    </div>
</div>
